package ru.yandex.solomon.experiments.gordiychuk;

import java.time.Clock;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.TearDown;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.dumper.ConcurrentMetricsCache;
import ru.yandex.solomon.dumper.MetricImpl;
import ru.yandex.solomon.dumper.MetricsCacheMetrics;
import ru.yandex.solomon.dumper.TickClock;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 20, batchSize = 30)
@Warmup(iterations = 10, batchSize = 30)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.NANOSECONDS)
public class ConcurrentCacheJmh {
    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(ConcurrentCacheJmh.class.getName())
                .detectJvmArgs()
//                .addProfiler(FlightRecorderProfiler.class)
                .build();

        new Runner(opt).run();
    }

        /*

Test with Caffaine base cache
Benchmark                    Mode  Cnt     Score     Error  Units
ConcurrentCacheJmh.baseLine  avgt   20   768.756 ±  47.488  ns/op
ConcurrentCacheJmh.hit       avgt   20  2872.844 ± 175.401  ns/op
ConcurrentCacheJmh.hitMap    avgt   20   504.787 ±   4.817  ns/op
ConcurrentCacheJmh.miss      avgt   20   527.017 ±   4.456  ns/op
ConcurrentCacheJmh.missMap   avgt   20   482.108 ±   3.811  ns/op

Benchmark                       Mode  Cnt     Score    Error  Units
ConcurrentCacheJmh.baseLine     avgt   20   756.047 ± 14.318  ns/op
ConcurrentCacheJmh.hit          avgt   20   612.286 ± 26.759  ns/op
ConcurrentCacheJmh.hitMap       avgt   20   590.627 ± 11.954  ns/op
ConcurrentCacheJmh.hitOpenMap   avgt   20  1043.587 ±  9.831  ns/op
ConcurrentCacheJmh.miss         avgt   20   580.365 ± 38.014  ns/op
ConcurrentCacheJmh.missMap      avgt   20   528.722 ± 21.013  ns/op
ConcurrentCacheJmh.missOpenMap  avgt   20  1038.673 ± 43.612  ns/op
     */

    private Labels miss;
    private Labels hit;
    private ConcurrentMetricsCache cache;
    private ConcurrentHashMap<Labels, MetricImpl> map;
    private Object2ObjectOpenHashMap<Labels, MetricImpl> openMap;
    private ReadWriteLock rwLock;
    private ScheduledExecutorService timer;
    private TickClock clock;

    @Setup
    public void prepare() {
        timer = Executors.newSingleThreadScheduledExecutor();
        clock = new TickClock(Clock.systemUTC());
        cache = new ConcurrentMetricsCache(ForkJoinPool.commonPool(), timer, clock, new MetricsCacheMetrics(new MetricRegistry()));
        map = new ConcurrentHashMap<>();
        openMap = new Object2ObjectOpenHashMap<>();
        rwLock = new ReentrantReadWriteLock();
        hit = Labels.of("name", "alice", "cpu", "01", "dc", "man", "host", "test");
        miss = Labels.of("name", "bob", "cpu", "01", "dc", "man", "host", "test");
        cache.add(Map.of(hit, new MetricImpl(42, 101, MetricType.DGAUGE)));
        map.putAll(Map.of(hit, new MetricImpl(42, 101, MetricType.DGAUGE)));
        openMap.putAll(Map.of(hit, new MetricImpl(42, 101, MetricType.DGAUGE)));
    }

    @TearDown
    public void tearDown() {
        cache.close();
        timer.shutdownNow();
    }

    @Benchmark
    public Object hit() {
        return cache.resolve(hit);
    }

    @Benchmark
    public Object miss() {
        return cache.resolve(miss);
    }

    @Benchmark
    public Object hitMap() {
        var r = map.get(hit);
        r.touch(clock.millis());
        return r;
    }

    @Benchmark
    public Object missMap() {
        return map.get(miss);
    }

    @Benchmark
    public Object hitOpenMap() {
        rwLock.readLock().lock();
        try {
            var r = openMap.get(hit);
            r.touch(clock.millis());
            return r;
        } finally {
            rwLock.readLock().unlock();
        }
    }

    @Benchmark
    public Object missOpenMap() {
        rwLock.readLock().lock();
        try {
            return map.get(miss);
        } finally {
            rwLock.readLock().unlock();
        }
    }

    @Benchmark
    public long baseLine() {
        return System.currentTimeMillis();
    }
}
