package ru.yandex.solomon.experiments.gordiychuk;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import com.google.common.collect.Lists;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.table.description.TableColumn;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.Type;
import com.yandex.ydb.table.values.Value;

import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

/**
 * Copy one table into another one, helpful for extend primary key.
 *
 * @author Sergey Polovko
 */
public class CopyTable {

    private static final class TableData {
        final StructType rowType;
        final List<Value> rows = new ArrayList<>();

        TableData(TableDescription tableDesc) {
            List<TableColumn> columns = tableDesc.getColumns();
            Map<String, Type> members = new HashMap<>(columns.size());
            for (TableColumn column : columns) {
                members.put(column.getName(), column.getType());
            }
            this.rowType = StructType.of(members);
        }

        void addRow(ResultSetReader resultSet) {
            var fields = new HashMap<String, Value>(resultSet.getColumnCount());
            for (int i = 0; i < resultSet.getColumnCount(); i++) {
                fields.put(resultSet.getColumnName(i), resultSet.getColumn(i).getValue());
            }
            rows.add(rowType.newValue(fields));
        }
    }

    private static TableDescription describeTable(YdbClient client, String path) {
        return client.fluent().describeTable(path)
                .join()
                .expect("cannot describe table " + path);
    }

    private static TableData readAll(YdbClient client, String path, TableDescription tableDesc) {
        var settings = ReadTableSettings.newBuilder()
                .timeout(2, TimeUnit.MINUTES)
                .orderedRead(true)
                .build();

        return client.fluent().callOnSession(session -> {
            TableData tableData = new TableData(tableDesc);
            return session.readTable(path, settings, resultSet -> {
                while (resultSet.next()) {
                    tableData.addRow(resultSet);
                }
            }).thenApply(s -> s.isSuccess() ? Result.success(tableData) : Result.fail(s));
        }).join().expect("cannot read table " + path);
    }

    private static void writeAll(YdbClient client, String destPath, TableData data) {
        ListType rowsType = ListType.of(data.rowType);
        String query = String.format("""
                --!syntax_v1
                DECLARE $rows AS %s;
                REPLACE INTO `%s` SELECT * FROM AS_TABLE($rows);
                """, rowsType, destPath);

        var it = Lists.partition(data.rows, 1000).iterator();
        AtomicInteger counter = new AtomicInteger();
        AsyncActorBody body = () -> {
            while (true) {
                if (!it.hasNext()) {
                    return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                }

                var partition = it.next();
                Params params = Params.of("$rows", rowsType.newValue(partition));
                return client.fluent().execute(query, params)
                        .thenApply(r -> {
                            System.out.println(counter.addAndGet(partition.size()) + " copy");
                            return r.expect("cannot insert data into " + destPath);
                        });
            }
        };
        var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 30);
        runner.start().join();
    }

    public static void copyTable(YdbClient ydb, String srcPath, String dstPath) {
        var tableDesc = describeTable(ydb, srcPath);
        var read = readAll(ydb, srcPath, tableDesc);
        writeAll(ydb, dstPath, read);
    }
}
