package ru.yandex.solomon.experiments.gordiychuk;

import java.util.Collections;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import it.unimi.dsi.fastutil.ints.IntArrayList;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.gateway.tasks.removeShard.RemoveShardTaskHandler;
import ru.yandex.solomon.scheduler.dao.SchedulerDao;
import ru.yandex.solomon.scheduler.dao.ydb.YdbSchedulerDao;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.cleanup.NumIdResolver;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class DeleteLostShards implements AutoCloseable {

    private final SolomonCluster cluster;
    private final YdbClient ydb;
    private final SchedulerDao schedulerDao;

    public DeleteLostShards(SolomonCluster cluster) {
        this.cluster = cluster;
        this.ydb = YdbHelper.createYdbClient(cluster);
        schedulerDao = new YdbSchedulerDao(cluster.kikimrRootPath() + "/Gateway/V1/TaskScheduler", ydb.table, ydb.scheme);
    }

    public static void main(String[] args) {
        try (var cli = new DeleteLostShards(SolomonCluster.CLOUD_PROD_FRONT)) {
            cli.scheduleRemove();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
        System.exit(0);
    }

    public void scheduleRemove() {
        var numIdToShardId = NumIdResolver.numIdToShardId(cluster);
        var numIdToShard = NumIdResolver.numIdToShard(cluster);

        var toDelete = new IntArrayList();
        for (var entry : numIdToShardId.int2ObjectEntrySet()) {
            if (!numIdToShard.containsKey(entry.getIntKey())) {
                toDelete.add(entry.getIntKey());
            }
        }
        Collections.shuffle(toDelete);

        var it = toDelete.iterator();
        AtomicLong shards = new AtomicLong();
        AtomicInteger processed = new AtomicInteger();
        AsyncActorBody body = () -> {
            if (!it.hasNext()) {
                return completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var numId = it.nextInt();
            var shardId = numIdToShardId.get(numId);

            reportProgress(processed.incrementAndGet(), toDelete.size());
            System.out.println("Processed shards " + DataSize.shortString(shards.get()));
            return scheduleDelete(numId, shardId)
                    .thenAccept((count) -> shards.incrementAndGet());
        };

        var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 10);
        runner.start().join();
        System.out.println("Done!");
    }

    private void reportProgress(int count, long total) {
        double progress = count * 100. / total;
        System.out.println("Progress " + String.format("%.2f%%", progress));
    }

    private CompletableFuture<?> scheduleDelete(int numId, String shardId) {
        var jitter = ThreadLocalRandom.current().nextLong(TimeUnit.DAYS.toMillis(30L));
        var task = RemoveShardTaskHandler.removeShardTask(projectId(shardId), shardId, numId)
                .toBuilder()
                .setExecuteAt(System.currentTimeMillis() + jitter)
                .build();

        return schedulerDao.add(task).thenRun(() ->
                System.out.println("schedule remove" +
                        " numId:" + Integer.toUnsignedLong(numId)
                        + " shardId: " + shardId
                        + " taskId:" + task.id()));
    }

    private String projectId(String shardId) {
        var parts = StringUtils.split(shardId, '_');
        if (parts.length <= 1) {
            return "";
        }
        return parts[0];
    }

    @Override
    public void close() {
        ydb.close();
    }
}
