package ru.yandex.solomon.experiments.gordiychuk;

import java.nio.charset.StandardCharsets;
import java.time.Clock;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.hash.Hashing;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.DistributedLockImpl;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.locks.dao.ydb.YdbLocksDao;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;


/**
 * @author Vladimir Gordiychuk
 */
public class DistributedKikimrLock {
    private static final Logger logger = LoggerFactory.getLogger(DistributedKikimrLock.class);
    private static final Clock CLOCK = Clock.systemUTC();
    private static final String LOCK_ID = "test-leader";

    public static void main(String[] args) {
        ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(Runtime.getRuntime().availableProcessors());
        String path = "/Root/Solomon/Alerting/Locks3Sec";

        YdbClient ydbClient = YdbHelper.createYdbClient(SolomonCluster.PRESTABLE_STORAGE);
        ydbClient.scheme.makeDirectories(path).join().expect("cannot create dir " + path);

        YdbLocksDao dao = new YdbLocksDao(ydbClient.table, CLOCK, path);
        dao.createSchema().join();

        List<DistributedLock> locks = IntStream.range(0, 100)
                .mapToObj(index -> makeLock(index, dao, scheduler))
                .collect(Collectors.toList());
        logger.info("All locks created, waiting");
    }

    private static DistributedLock makeLock(int index, LocksDao dao, ScheduledExecutorService scheduler) {
        String appId = Hashing.sha1()
                .hashString("host-" + index, StandardCharsets.UTF_8)
                .toString();

        DistributedLock lock = new DistributedLockImpl(LOCK_ID, LOCK_ID, appId, dao, CLOCK, scheduler, new MetricRegistry());
        acquireForever(lock);
        return lock;
    }

    private static void acquireForever(DistributedLock lock) {
        lock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return false;
            }

            @Override
            public void onLock(long seqNo) {
                logger.info("Distributed Lock Locked: {}", seqNo);
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                logger.info("Distributed Lock unlocked by " + reason);
                acquireForever(lock);
            }
        }, 3, TimeUnit.SECONDS);
    }
}
