package ru.yandex.solomon.experiments.gordiychuk;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.infra.Blackhole;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 20)
@Warmup(iterations = 10)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.Throughput)
@OutputTimeUnit(TimeUnit.SECONDS)
public class EventDispatchJmh {
    private List<Event> events;
    private Handler handler;

    /*

Benchmark                             Mode  Cnt       Score      Error  Units
EventDispatchJmh.dispatchEnumSwitch  thrpt   20  238308.931 ± 2820.544  ops/s
EventDispatchJmh.dispatchInstanceOf  thrpt   20   78562.161 ±  288.712  ops/s
EventDispatchJmh.dispatchVisitor     thrpt   20  253297.348 ±  874.063  ops/s

     */

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(EventDispatchJmh.class.getName())
                .detectJvmArgs()
//                .addProfiler(AsyncProfiler.class)
                .build();

        new Runner(opt).run();
    }

    @Setup
    public void prepare(Blackhole bh) {
        handler = new HandlerImpl(bh);
        events = new ArrayList<>();
        for (int index = 0; index < 3; index++) {
            events.add(new AssignEventImpl(index));
        }

        for (int index = 0; index < 3; index++) {
            events.add(new ReleaseEventImpl(index));
        }

        for (int index = 0; index < 3; index++) {
            events.add(new DestroyEventImpl(index));
        }

        for (int index = 0; index < 100; index++) {
            events.add(new CommitEventImpl(index));
            events.add(new DataEventImpl(index));
        }

        Collections.shuffle(events);
    }

    @Benchmark
    public void dispatchInstanceOf() {
        for (var event : events) {
            if (event instanceof DataEvent) {
                handler.onDataEvent((DataEvent) event);
            } else if (event instanceof CommitEvent) {
                handler.onCommitEvent((CommitEvent) event);
            } else if (event instanceof AssignEvent) {
                handler.onAssignEvent((AssignEvent) event);
            } else if (event instanceof ReleaseEvent) {
                handler.onReleaseEvent((ReleaseEvent) event);
            } else if (event instanceof DestroyEvent) {
                handler.onDestroyEvent((DestroyEvent) event);
            }
        }
    }

    @Benchmark
    public void dispatchVisitor() {
        for (var event : events) {
            event.visit(handler);
        }
    }

    @Benchmark
    public void dispatchEnumSwitch() {
        for (var event : events) {
            switch (event.type()) {
                case DATA:
                    handler.onDataEvent((DataEvent) event);
                    break;
                case COMMIT:
                    handler.onCommitEvent((CommitEvent) event);
                    break;
                case ASSIGN:
                    handler.onAssignEvent((AssignEvent) event);
                    break;
                case RELEASE:
                    handler.onReleaseEvent((ReleaseEvent) event);
                    break;
                case DESTROY:
                    handler.onDestroyEvent((DestroyEvent) event);
                    break;
                default:
                    throw new UnsupportedOperationException("unsupported type: " + event.type());
            }
        }
    }

    enum EventType {
        DATA, ASSIGN, RELEASE, DESTROY, COMMIT
    }

    interface Event {
        void visit(Handler handler);
        EventType type();
    }

    interface DataEvent extends Event {
        int getData();
    }

    interface AssignEvent extends Event {
        int getAssign();
    }

    interface ReleaseEvent extends Event {
        int getRelease();
    }

    interface DestroyEvent extends Event {
        int getDestroy();
    }

    interface CommitEvent extends Event {
        int getCommit();
    }

    private static class DataEventImpl implements DataEvent {
        private final int data;

        public DataEventImpl(int data) {
            this.data = data;
        }

        @Override
        public int getData() {
            return data;
        }

        @Override
        public EventType type() {
            return EventType.DATA;
        }

        @Override
        public void visit(Handler handler) {
            handler.onDataEvent(this);
        }
    }

    private static class AssignEventImpl implements AssignEvent {
        private final int assign;

        public AssignEventImpl(int assign) {
            this.assign = assign;
        }

        @Override
        public int getAssign() {
            return assign;
        }

        @Override
        public EventType type() {
            return EventType.ASSIGN;
        }

        @Override
        public void visit(Handler handler) {
            handler.onAssignEvent(this);
        }
    }

    private static class ReleaseEventImpl implements ReleaseEvent {
        private final int release;

        public ReleaseEventImpl(int release) {
            this.release = release;
        }

        @Override
        public int getRelease() {
            return release;
        }

        @Override
        public EventType type() {
            return EventType.RELEASE;
        }

        @Override
        public void visit(Handler handler) {
            handler.onReleaseEvent(this);
        }
    }

    private static class DestroyEventImpl implements DestroyEvent {
        private final int destroy;

        public DestroyEventImpl(int destroy) {
            this.destroy = destroy;
        }

        @Override
        public int getDestroy() {
            return destroy;
        }

        @Override
        public EventType type() {
            return EventType.DESTROY;
        }

        @Override
        public void visit(Handler handler) {
            handler.onDestroyEvent(this);
        }
    }

    private static class CommitEventImpl implements CommitEvent {
        private final int commit;

        public CommitEventImpl(int commit) {
            this.commit = commit;
        }

        @Override
        public int getCommit() {
            return commit;
        }

        @Override
        public EventType type() {
            return EventType.COMMIT;
        }

        @Override
        public void visit(Handler handler) {
            handler.onCommitEvent(this);
        }
    }

    interface Handler {
        void onDataEvent(DataEvent event);

        void onAssignEvent(AssignEvent event);

        void onReleaseEvent(ReleaseEvent event);

        void onDestroyEvent(DestroyEvent event);

        void onCommitEvent(CommitEvent event);
    }

    private class HandlerImpl implements Handler{
        private final Blackhole bh;

        public HandlerImpl(Blackhole bh) {
            this.bh = bh;
        }

        @Override
        public void onDataEvent(DataEvent event) {
            bh.consume(event.getData());
        }

        @Override
        public void onAssignEvent(AssignEvent event) {
            bh.consume(event.getAssign());
        }

        @Override
        public void onReleaseEvent(ReleaseEvent event) {
            bh.consume(event.getRelease());
        }

        @Override
        public void onDestroyEvent(DestroyEvent event) {
            bh.consume(event.getDestroy());
        }

        @Override
        public void onCommitEvent(CommitEvent event) {
            bh.consume(event.getCommit());
        }
    }


}
