package ru.yandex.solomon.experiments.gordiychuk;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Group;
import org.openjdk.jmh.annotations.GroupThreads;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.monlib.metrics.encode.text.MetricTextEncoder;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.server.shard.stat.GlobalUsageStats;

import static ru.yandex.solomon.model.point.AggrPointDataTestSupport.randomPoint;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 5, time = 2, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 2, time = 2, timeUnit = TimeUnit.SECONDS)
@State(Scope.Group)
@BenchmarkMode(Mode.AverageTime)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
public class GlobalUsageStatJmh {

    @Param({"1", "10", "100"})
    private int uniqueShards;

    private GlobalUsageStats stats;
    private EProjectId projectId;
    private List<MetricArchiveMutable> archives;

    /*

BEFORE:
Benchmark                            (uniqueShards)  Mode  Cnt  Score   Error  Units
GlobalUsageStatJmh.g                              1  avgt    5  0.839 ± 0.280  ms/op
GlobalUsageStatJmh.g:collect                      1  avgt    5  0.045 ± 0.004  ms/op
GlobalUsageStatJmh.g:updateOneByOne               1  avgt    5  1.633 ± 0.557  ms/op
GlobalUsageStatJmh.g                             10  avgt    5  1.096 ± 0.404  ms/op
GlobalUsageStatJmh.g:collect                     10  avgt    5  0.178 ± 0.043  ms/op
GlobalUsageStatJmh.g:updateOneByOne              10  avgt    5  2.015 ± 0.768  ms/op
GlobalUsageStatJmh.g                            100  avgt    5  2.171 ± 0.741  ms/op
GlobalUsageStatJmh.g:collect                    100  avgt    5  1.707 ± 0.687  ms/op
GlobalUsageStatJmh.g:updateOneByOne             100  avgt    5  2.636 ± 0.826  ms/op

NOW:
Benchmark                            (uniqueShards)  Mode  Cnt  Score   Error  Units
GlobalUsageStatJmh.bulk                           1  avgt    5  0.096 ± 0.036  ms/op
GlobalUsageStatJmh.bulk:bulkUpdate                1  avgt    5  0.155 ± 0.067  ms/op
GlobalUsageStatJmh.bulk:collectBulk               1  avgt    5  0.037 ± 0.006  ms/op
GlobalUsageStatJmh.bulk                          10  avgt    5  0.199 ± 0.107  ms/op
GlobalUsageStatJmh.bulk:bulkUpdate               10  avgt    5  0.211 ± 0.141  ms/op
GlobalUsageStatJmh.bulk:collectBulk              10  avgt    5  0.187 ± 0.082  ms/op
GlobalUsageStatJmh.bulk                         100  avgt    5  1.017 ± 0.378  ms/op
GlobalUsageStatJmh.bulk:bulkUpdate              100  avgt    5  0.266 ± 0.079  ms/op
GlobalUsageStatJmh.bulk:collectBulk             100  avgt    5  1.768 ± 0.681  ms/op
GlobalUsageStatJmh.g                              1  avgt    5  0.227 ± 0.035  ms/op
GlobalUsageStatJmh.g:collect                      1  avgt    5  0.031 ± 0.011  ms/op
GlobalUsageStatJmh.g:updateOneByOne               1  avgt    5  0.423 ± 0.069  ms/op
GlobalUsageStatJmh.g                             10  avgt    5  0.391 ± 0.101  ms/op
GlobalUsageStatJmh.g:collect                     10  avgt    5  0.208 ± 0.069  ms/op
GlobalUsageStatJmh.g:updateOneByOne              10  avgt    5  0.574 ± 0.136  ms/op
GlobalUsageStatJmh.g                            100  avgt    5  1.191 ± 0.310  ms/op
GlobalUsageStatJmh.g:collect                    100  avgt    5  1.613 ± 0.484  ms/op
GlobalUsageStatJmh.g:updateOneByOne             100  avgt    5  0.768 ± 0.217  ms/op

     */

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(GlobalUsageStatJmh.class.getName())
            .detectJvmArgs()
            .build();

        new Runner(opt).run();
    }

    @Setup
    public void setUp() {
        stats = new GlobalUsageStats();
        projectId = EProjectId.SOLOMON;
        archives = new ArrayList<>(10_000);
        for (int index =0 ; index < 10_000; index++) {
            MetricArchiveMutable archive = new MetricArchiveMutable();
            archive.setType(MetricType.DGAUGE);
            archive.addRecord(randomPoint(MetricType.DGAUGE));
            archive.setOwnerProjectIdEnum(projectId);
            archive.setOwnerShardId(uniqueShards == 0 ? 42 : ThreadLocalRandom.current().nextInt(0, uniqueShards)); // high contention
            archives.add(archive);
        }
    }

    @Benchmark
    @Group("g")
    @GroupThreads(2)
    public Object updateOneByOne() {
        for (var archive : archives) {
            stats.write(archive);
        }
        return stats;
    }

    @Benchmark
    @Group("g")
    @GroupThreads(2)
    public Object collect() {
        StringWriter writer = new StringWriter();
        MetricTextEncoder text = new MetricTextEncoder(writer, true);
        stats.stats.supply(0, text);
        return writer.toString();
    }

    @Benchmark
    @Group("bulk")
    @GroupThreads(2)
    public Object bulkUpdate() {
        stats.write(archives);
        return stats;
    }

    @Benchmark
    @Group("bulk")
    @GroupThreads(2)
    public Object collectBulk() {
        StringWriter writer = new StringWriter();
        MetricTextEncoder text = new MetricTextEncoder(writer, true);
        stats.stats.supply(0, text);
        return writer.toString();
    }
}
