package ru.yandex.solomon.experiments.gordiychuk;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import com.google.common.net.HostAndPort;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.ShardSelectors;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.TSliceOptions;
import ru.yandex.solomon.metrics.client.MetabaseStatus;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.cfg.SolomonPorts;
import ru.yandex.solomon.tool.cleanup.NumIdResolver;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.metrics.client.ResponseValidationUtils.ensureMetabaseStatusValid;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricBySelectorCheck implements AutoCloseable {
    private static final InterningLabelAllocator ALLOCATOR = new InterningLabelAllocator();
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withDelay(1_000)
            .withMaxDelay(60_000)
            .withNumRetries(10)
            .withStats((timeSpentMillis, cause) -> cause.printStackTrace());

    private final MetabaseClient metabase;

    public MetricBySelectorCheck(SolomonCluster cluster) {
        this.metabase = createMetabaseClient(cluster);
    }

    /**
     * ./run.sh ru.yandex.solomon.experiments.gordiychuk.MetricBySelectorCheck <CLUSTER> <SELECTOR> <SHOW_METRICS>
     * ./run.sh -j /usr/local/jdk-15/bin/java --enable-preview ru.yandex.solomon.experiments.gordiychuk.MetricBySelectorCheck CLOUD_PROD_FETCHER_SAS '{project=solomon, cluster="dev", service="sys", path="/Filesystem/FreeB"}'
     * @param args
     */
    public static void main(String[] args) {
        try {
            var cluster = SolomonCluster.valueOf(args[0]);
            var selectors = Selectors.parse(args[1]);
            var showMetrics = args.length >= 3 && Boolean.parseBoolean(args[2]);
            try (var task = new MetricBySelectorCheck(cluster)) {
                System.out.println("Init metabase...");
                task.metabase.forceUpdateClusterMetaData().completeOnTimeout(null, 5, TimeUnit.MINUTES).join();

                System.out.println("Fetch shards...");
                var shards = NumIdResolver.numIdToShard(cluster);

                task.find(shards, selectors, showMetrics);
            }
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    public void find(Int2ObjectMap<Shard> shardByNumId, Selectors selectors, boolean showMetrics) {
        var shardSelectors = ShardSelectors.onlyShardKey(selectors);
        var shards = shardByNumId
                .values()
                .stream()
                .filter(shard -> {
                    var key = ShardKey.create(shard.getProjectId(), shard.getClusterName(), shard.getServiceName());
                    return shardSelectors.match(key);
                })
                .collect(Collectors.toList());

        int batchSize = Math.max((int) Math.round(shards.size() * 0.01), 1);

        ConcurrentLinkedQueue<ResultMatch> results = new ConcurrentLinkedQueue<>();
        AtomicInteger completed = new AtomicInteger();
        var it = shards.iterator();
        AsyncActorBody body = () -> {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var shard = it.next();
            return find(shard, selectors)
                    .thenAccept(results::add)
                    .thenRun(() -> {
                        int cnt = completed.incrementAndGet();
                        if (cnt % batchSize == 0) {
                            double progress = cnt * 100. / shards.size();
                            System.out.println("Progress " + String.format("%.2f%%", progress));
                        }
                    });
        };

        var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 100);
        runner.start().join();

        var presentList = results.stream()
                .filter(ResultMatch::present)
                .collect(Collectors.toList());

        if (showMetrics) {
            System.out.println();
            System.out.println("Metric present in shards:");
            System.out.println("====");
            for (var result : presentList) {
                System.out.println("numId: " + Integer.toUnsignedString(result.shard.getNumId())
                        + ", projectId: " + result.shard.getProjectId()
                        + ", cluster: " + result.shard.getClusterName()
                        + ", service: " +  result.shard.getServiceName()
                        + ", metric: " + LabelsFormat.format(result.labels)
                );
            }
        }

        System.out.println();
        System.out.println("Statistics: " + Selectors.format(selectors));
        System.out.println("=====");
        printCount("Matched shards: ", shards.size());
        printCount("Matched projects: ", shards.stream().map(Shard::getProjectId).distinct().count());
        printCount("Present in shards: ", presentList.size());
        printCount("Present in projects: ", presentList.stream().map(ResultMatch::shard).map(Shard::getProjectId).distinct().count());
    }

    private static void printCount(String prefix, long count) {
        var shortCount = DataSize.shortString(count);
        if (shortCount.equals(Long.toString(count))) {
            System.out.println(prefix + shortCount);
            return;
        }

        System.out.println(prefix + shortCount + " (" + count + ")");
    }

    private CompletableFuture<ResultMatch> find(Shard shard, Selectors selectors) {
        var selectorsWithShard = selectors.toBuilder()
                .add("project", shard.getProjectId())
                .add("cluster", shard.getClusterName())
                .add("service", shard.getServiceName())
                .build();

        return RetryCompletableFuture.runWithRetries(() -> {
            var req = FindRequest.newBuilder()
                    .addAllSelectors(LabelSelectorConverter.selectorsToProto(selectorsWithShard))
                    .setDeadlineMillis(System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(3))
                    .setSliceOptions(TSliceOptions.newBuilder()
                            .setLimit(1)
                            .build())
                    .build();

            return metabase.find(req)
                    .thenApply(response -> {
                        var status = MetabaseStatus.fromCode(response.getStatus(), response.getStatusMessage());
                        ensureMetabaseStatusValid(status);

                        if (response.getMetricsList().isEmpty()) {
                            return new ResultMatch(shard, false, Labels.of());
                        }

                        var labels = LabelConverter.protoToLabels(response.getMetricsList().get(0).getLabelsList(), ALLOCATOR);
                        return new ResultMatch(shard, true, labels);
                    });
        }, RETRY_CONFIG);
    }

    private static MetabaseClient createMetabaseClient(SolomonCluster cluster) {
        List<HostAndPort> addresses = cluster.hosts()
                .stream()
                .map(s -> HostAndPort.fromParts(s, SolomonPorts.COREMON_GRPC))
                .collect(toList());

        var options = MetabaseClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                        .setRequestTimeOut(1, TimeUnit.MINUTES)
                        .setKeepAliveDelay(1, TimeUnit.MINUTES)
                        .setKeepAliveTimeout(30, TimeUnit.SECONDS)
                        .setMaxInboundMessageSizeInBytes(200 << 20) // 200 Mib
                        .setMaxOutboundMessageSizeInBytes(200 << 20)) // 200 Mib
                .setExpireClusterMetadata(30, TimeUnit.SECONDS)
                .build();

        return MetabaseClients.create(addresses, options);
    }

    @Override
    public void close() throws Exception {
        metabase.close();
    }

    private static record ResultMatch(Shard shard, boolean present, Labels labels) {

    }
}
