package ru.yandex.solomon.experiments.gordiychuk;

import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.ByteString;
import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import org.apache.logging.log4j.Level;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.main.logger.LoggerConfigurationUtils;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.DeleteManyRequest;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindRequest;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.TSliceOptions;
import ru.yandex.solomon.metrics.client.TimeSeriesCodec;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.TimeSeries;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TCompressedWriteRequest;
import ru.yandex.stockpile.api.TReadRequest;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.tool.cfg.SolomonPorts.COREMON_GRPC;
import static ru.yandex.solomon.tool.cfg.SolomonPorts.STOCKPILE_GRPC;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricTypeMigrate implements AutoCloseable {
    private static final int IN_FLIGHT = 10;
    private static final Map<String, String> SERVICE_FROM_TO = ImmutableMap.<String, String>builder()
            .put("coremon_spack", "coremon")
            .put("coremon", "coremon")
            .put("stockpile", "stockpile")
            .put("stockpile_spack", "stockpile")
            .put("fetcher_spack", "fetcher")
            .put("fetcher_prod", "fetcher")
            .put("alerting", "alerting")
            .build();

    private static final Map<String, String> CLUSTER_FROM_TO = ImmutableMap.of(
            "fetcher_prod", "vlgo",
            "alerting_prod", "vlgo",
            "kikimr_solomon_stp_man", "vlgo",
            "kikimr_solomon_stp_myt", "vlgo");

    private final Unit from;
    private final Unit to;

    public MetricTypeMigrate() {
        Unit pre = new Unit(
            "conductor_group://solomon_pre_data_storage:" + STOCKPILE_GRPC,
            "conductor_group://solomon_pre_meta_storage:" + COREMON_GRPC);
        Unit prod = new Unit(
            "conductor_group://solomon_prod_data_storage_myt:" + STOCKPILE_GRPC,
            "conductor_group://solomon_prod_fetcher_myt:" + COREMON_GRPC);
        this.from = prod;
        this.to = pre;
    }

    public static void main(String[] args) {
        LoggerConfigurationUtils.simpleLogger(Level.INFO);
        try (MetricTypeMigrate migration = new MetricTypeMigrate()) {
            migration.awaitReady();

//            migration.migrateMeta("project=solomon, cluster=alerting_prod, alertType=total, service=alerting, bin='*', host=cluster, sensor=evaluations.eval.lagMillis, projectId=total").join();
//            migration.migrateMeta("project=solomon, cluster='fetcher_prod', service=coremon, path='/Pushes', shard_id=total, host=Man").join();
            migration.migrateMeta("project=solomon, cluster=kikimr_solomon_stp_myt, service=stockpile, path='/writeRecordResponseTimeMillisHistogram', bin='*', host=cluster").join();
//            migration.migrateMeta("project=solomon, cluster=alerting_prod, service=alerting, bin=*, sensor=grpc.client.call.elapsedTimeMs, host=cluster").join();
        } catch (Exception e) {
            e.printStackTrace();
        }
        System.exit(0);
    }

    private void awaitReady() {
        from.awaitReady();
        to.awaitReady();
    }

    private CompletableFuture<Void> migrateMeta(String selector) {
        Selectors selectors = Selectors.parse(selector);
        System.out.println("Starting migrate: " + selector);
        return migrateMeta(selectors, 0);
    }

    private CompletableFuture<Void> migrateMeta(Selectors selectors, int offset) {
        return from.metabase.find(FindRequest.newBuilder()
                .addAllSelectors(LabelSelectorConverter.selectorsToProto(selectors))
                .setSliceOptions(TSliceOptions.newBuilder()
                        .setOffset(offset)
                        .setLimit(IN_FLIGHT)
                        .build())
                .build())
                .thenCompose(response -> {
                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        throwInvalid(response);
                    }

                    System.out.println("Process offset: " + offset + " fetched " + response.getMetricsCount());
                    return response.getMetricsList()
                            .parallelStream()
                            .map(this::migrateMetric)
                            .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                            .thenCompose(list -> {
                                if (list.size() < IN_FLIGHT) {
                                    return completedFuture(null);
                                }

                                return migrateMeta(selectors, offset + list.size());
                            });
                });
    }

    private CompletableFuture<Void> migrateMetric(Metric fromMetric) {
        if (fromMetric.getType() == MetricType.DGAUGE) {
            System.out.println("Skip: " + TextFormat.shortDebugString(fromMetric));
            return completedFuture(null);
        }

        return resolveOrCreate(fromMetric)
                .thenCompose(toMetric -> migrateMetricData(fromMetric, toMetric));
    }

    private CompletableFuture<Void> migrateMetricData(Metric fromMetric, Metric toMetric) {
        return read(fromMetric)
                .thenApply(source -> {
                    MetricArchiveMutable archive = new MetricArchiveMutable();
                    archive.setType(toMetric.getType());
                    archive.addAllFrom(new ToStringIterator(MetricTypeTransfers.of(MetricType.DGAUGE, toMetric.getType(), new ToStringIterator(source.iterator(), "i")), "o"));
                    return archive;
                })
                .thenCompose(archive -> save(toMetric, archive));
    }

    private CompletableFuture<AggrGraphDataIterable> read(Metric metric) {
        System.out.println("Start reading metric: "+ TextFormat.shortDebugString(metric));
        StockpileFormat format = StockpileFormat.byNumberOrCurrent(from.stockpile.getCompatibleCompressFormat().upperEndpoint());
        return from.stockpile.readCompressedOne(TReadRequest.newBuilder()
                .setMetricId(metric.getMetricId())
                .setBinaryVersion(format.getFormat())
                .build())
                .thenApply(response -> {
                    if (response.getStatus() != EStockpileStatusCode.OK) {
                        throwInvalid(response);
                    }


                    AggrGraphDataIterable result = TimeSeriesCodec.sequenceDecode(response);
                    System.out.println("Read points " + result.getRecordCount() + " by metric " + TextFormat.shortDebugString(metric));
                    return result;
                });
    }

    private CompletableFuture<Void> save(Metric metric, MetricArchiveMutable archive) {
        System.out.println("Save metric " + TextFormat.shortDebugString(metric) + " points " + archive.getRecordCount());
        ByteString data = MetricArchiveNakedSerializer.serializerForFormatSealed(StockpileFormat.CURRENT)
                .serializeToByteString(archive.toImmutableNoCopy());

        TimeSeries.Chunk chunk = TimeSeries.Chunk.newBuilder()
                .setContent(data)
                .setPointCount(archive.getRecordCount())
                .build();

        return to.stockpile.writeCompressedOne(TCompressedWriteRequest.newBuilder()
                .setMetricId(metric.getMetricId())
                .setBinaryVersion(StockpileFormat.CURRENT.getFormat())
                .addChunks(chunk)
                .build())
                .thenAccept(response -> {
                    if (response.getStatus() != EStockpileStatusCode.OK) {
                        throwInvalid(response);
                    }
                });
    }

    private CompletableFuture<Metric> resolve(Labels labels) {
        return to.metabase.resolveOne(ResolveOneRequest.newBuilder()
                .addAllLabels(LabelConverter.labelsToProtoList(labels))
                .build())
                .thenApply(response -> {
                    if (response.getStatus() == EMetabaseStatusCode.NOT_FOUND) {
                        return null;
                    }

                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        throwInvalid(response);
                    }

                    System.out.println("Metric already created into metabase: " + TextFormat.shortDebugString(response.getMetric()));
                    return response.getMetric();
                });
    }

    private CompletableFuture<Metric> create(Metric metric) {
        return to.metabase.createOne(CreateOneRequest.newBuilder()
                .setMetric(metric)
                .build())
                .thenApply(response -> {
                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        throwInvalid(response);
                    }

                    System.out.println("Created metric: " + TextFormat.shortDebugString(response.getMetric()));
                    return response.getMetric();
                });
    }

    private CompletableFuture<Metric> resolveOrCreate(Metric metric) {
        Labels from = LabelConverter.protoToLabels(metric.getLabelsList());
        Labels to = from.toBuilder()
                .remove("project")
                .remove("cluster")
                .remove("service")
                .add("project", "junk")
                .add("cluster", CLUSTER_FROM_TO.get(from.findByKey("cluster").getValue()))
                .add("service", SERVICE_FROM_TO.get(from.findByKey("service").getValue()))
                .build();

        return resolve(to)
                .thenCompose(resolved -> {
                    if (resolved != null) {
                        return delete(resolved);
                    }

                    return completedFuture(null);
                })
                .thenCompose(ignore -> create(metric.toBuilder()
                        .clearMetricId()
                        .clearLabels()
                        .addAllLabels(LabelConverter.labelsToProtoList(to))
                        .build()));
    }

    private CompletableFuture<Void> delete(Metric delete) {
        return to.metabase.deleteMany(DeleteManyRequest.newBuilder()
                .addMetrics(delete)
                .build())
                .thenAccept(response -> {
                    if (response.getStatus() != EMetabaseStatusCode.OK) {
                        throwInvalid(response);
                    }
                });
    }

    private void throwInvalid(Message message) {
        throw new IllegalStateException(TextFormat.shortDebugString(message));
    }

    @Override
    public void close() {
        from.close();
        to.close();
    }

    @ParametersAreNonnullByDefault
    private static class ToStringIterator extends AggrGraphDataListIterator {
        private AggrGraphDataListIterator source;
        private String prefix;

        public ToStringIterator(AggrGraphDataListIterator source, String prefix) {
            super(source.columnSetMask());
            this.source = source;
            this.prefix = prefix;
        }

        @Override
        public boolean next(AggrPoint target) {
            if (source.next(target)) {
                System.out.println(prefix + ": " + target);
                return true;
            }

            return false;
        }

        @Override
        public int estimatePointsCount() {
            return source.estimatePointsCount();
        }
    }

    private static class Unit implements AutoCloseable {
        private final StockpileClient stockpile;
        private final MetabaseClient metabase;

        public Unit(String stockpile, String metabase) {
            this.stockpile = createStockpileClient(stockpile);
            this.metabase = createMetabaseClient(metabase);
        }

        private StockpileClient createStockpileClient(String address) {
            var options = StockpileClientOptions.newBuilder(
                    DefaultClientOptions.newBuilder()
                            .setRequestTimeOut(5, TimeUnit.MINUTES)
                            .setMaxInboundMessageSizeInBytes(200 << 20)
                            .setMaxInboundMessageSizeInBytes(200 << 20))
                    .setExpireClusterMetadata(30, TimeUnit.SECONDS)
                    .setRetryStopStrategy(StopStrategies.stopAfterAttempt(20))
                    .build();

            return StockpileClients.createDynamic(List.of(address), options);
        }

        private MetabaseClient createMetabaseClient(String address) {
            var options = MetabaseClientOptions.newBuilder(
                    DefaultClientOptions.newBuilder()
                        .setRequestTimeOut(5, TimeUnit.MINUTES)
                        .setMaxInboundMessageSizeInBytes(200 << 20)
                        .setMaxInboundMessageSizeInBytes(200 << 20))
                    .setExpireClusterMetadata(30, TimeUnit.MINUTES)
                    .setMetaDataRequestTimeOut(30, TimeUnit.SECONDS)
                    .build();

            return MetabaseClients.createDynamic(List.of(address), options);
        }

        private void awaitReady() {
            while (stockpile.getReadyShardsCount() < 4096) {
                stockpile.forceUpdateClusterMetaData().join();
            }
        }

        @Override
        public void close() {
            stockpile.close();
            metabase.close();
        }
    }
}
