package ru.yandex.solomon.experiments.gordiychuk;

import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.netty.handler.codec.http.HttpResponseStatus;
import org.apache.commons.io.IOUtils;

import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.host.HostUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class PushToCloudPreprod {
    private static String CLOUD_ID = "aoecngvoh58bgtr3s25a";
//    private static String CLOUD_ID = "aoege385gppokk0mvs3u";
    private static String FOLDER_ID = "aoe6vrq0g3svvs3uf62u";
//    private static String FOLDER_ID = "aoedtuthhbonup8uq59m";
    private static String SERVICE = "custom";

    private volatile String token;
    private final ScheduledExecutorService timer;
    private final ObjectMapper mapper = new ObjectMapper();
    private final HttpClient httpClient = HttpClient.newBuilder()
        .connectTimeout(Duration.ofSeconds(30))
        .followRedirects(HttpClient.Redirect.NEVER)
        .version(HttpClient.Version.HTTP_1_1)
        .build();

    private final AtomicLong prev = new AtomicLong(0);
    private volatile ScheduledFuture task;

    public PushToCloudPreprod(ScheduledExecutorService timer) {
        this.token = getIamToken();
        this.timer = timer;
    }

    public static void main(String[] args) throws ExecutionException, InterruptedException {
        var timer = Executors.newScheduledThreadPool(1);
        PushToCloudPreprod proc = new PushToCloudPreprod(timer);
        proc.start();
        proc.task.get();
//        proc.pushToLimit(50_000_000);
        System.exit(0);
    }

    public void pushToLimit(long limit) {
        AtomicLong pushed = new AtomicLong();
        AtomicLong inFlight = new AtomicLong();

        AsyncActorBody body = () -> {
            if (pushed.get() >= limit && inFlight.get() == 0) {
                return completedFuture(AsyncActorBody.DONE_MARKER);
            }

            long capacity = limit - (pushed.get() + inFlight.get());
            if (capacity <= 0) {
                CompletableFuture<Void> future = new CompletableFuture<>();
                CompletableFuture.delayedExecutor(5, TimeUnit.SECONDS).execute(() -> future.complete(null));
                return future;
            }

            int size = (int) Math.min(capacity, 1000L);
            inFlight.addAndGet(size);
            return push(size)
                .thenAccept(success -> {
                    if (success) {
                        long v = pushed.addAndGet(size);
                        double progress = v * 100. / limit;
                        System.out.println("Progress: " + String.format("%.2f%%", progress));
                    }
                    inFlight.addAndGet(-size);
                });
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), Runtime.getRuntime().availableProcessors());
        runner.start().join();
    }

    private static String getIamToken() {
        try {
            Runtime r = Runtime.getRuntime();

            Process p = r.exec("/home/gordiychuk/yandex-cloud/bin/yc iam create-token --cloud-id " + CLOUD_ID + " --folder-id " + FOLDER_ID);
            String result = IOUtils.toString(p.onExit().join().getInputStream()).trim();
            p.destroy();
            return result;
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    public void start() {
        task = timer.scheduleAtFixedRate(this::push, 0, 1, TimeUnit.SECONDS);
    }

    public void stop() {
        task.cancel(false);
    }

    private CompletableFuture<Boolean> push() {
        return push(1);
    }

    private CompletableFuture<Boolean> push(int size) {
        try {
            var json = prepareJson(size);
//            System.out.println(json);
            var request = HttpRequest.newBuilder()
                .setHeader("Authorization", "Bearer " + token)
                .setHeader("Content-Type", "application/json")
                .POST(HttpRequest.BodyPublishers.ofString(json))
                .timeout(Duration.ofSeconds(10))
                .uri(URI.create("http://monitoring.api.cloud-preprod.yandex.net/monitoring/v1/data/write?cloud_id=" + CLOUD_ID + "&folder_id=" + FOLDER_ID + "&service=" + SERVICE))
                .build();

            return httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString())
                .thenApply(result -> {
                    System.out.println(result + " " + result.body());
                    if (result.statusCode() == HttpResponseStatus.UNAUTHORIZED.code()) {
                        token = getIamToken();
                    }

                    return result.statusCode() == HttpResponseStatus.OK.code();
                })
                .exceptionally(e -> {
                    e.printStackTrace();
                    return false;
                });
        } catch (Throwable e) {
            e.printStackTrace();
            return completedFuture(false);
        }
    }

    private String prepareJson(int size) {
        try {
            var now = Instant.now().truncatedTo(ChronoUnit.SECONDS);
            var timeseries = mapper.createArrayNode();
            for (int i = 0; i < size; i++) {
                long index = prev.incrementAndGet();
                double y = Math.sin(Math.toRadians(index));

                timeseries.add(mapper.createObjectNode()
                    .put("ts", now.plusSeconds(index).toString())
                    .put("value", y));
            }

            var metrics = mapper.createArrayNode();
            var metric = mapper.createObjectNode();
            metric.put("name", "sinusoid");
            metric.set("labels", mapper.createObjectNode().put("host", HostUtils.getFqdn()));
            metric.set("timeseries", timeseries);
            metrics.add(metric);

            var root = mapper.createObjectNode();
            root.set("metrics", metrics);

            return mapper.writeValueAsString(root);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }
}
