package ru.yandex.solomon.experiments.gordiychuk;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import com.google.protobuf.ByteString;
import it.unimi.dsi.fastutil.longs.Long2LongOpenHashMap;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.codec.serializer.StockpileDeserializer;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.IndexAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.IndexFile;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileActiveTimeseries implements AutoCloseable {
    private final SolomonCluster cluster;
    private final KikimrKvClient kvClient;

    public StockpileActiveTimeseries(SolomonCluster cluster) {
        this.cluster = cluster;
        this.kvClient = KikimrHelper.createKvClient(cluster);
    }

    public static void main(String[] args) {
        try (var task = new StockpileActiveTimeseries(SolomonCluster.PROD_STORAGE_VLA)) {
            task.run().join();
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(-1);
        }
        System.exit(0);
    }

    private CompletableFuture<?> run() {
        return resolveTablets()
                .thenCompose(localIds -> {
                    var inactiveTs = System.currentTimeMillis() - TimeUnit.HOURS.toMillis(4);
                    return statistics(localIds, inactiveTs);
                })
                .thenAccept(stats -> {
                    System.out.println("Active TimeSeries: " + DataSize.prettyString(stats.active) + "(" + stats.active +")");
                    System.out.println("Total TimeSeries : " + DataSize.prettyString(stats.count) + "(" + stats.count +")");
                    System.out.println("Active percent: " + stats.activePercent());
                });
    }

    private CompletableFuture<long[]> resolveTablets() {
        return kvClient.resolveKvTablets(cluster.getSolomonVolumePath());
    }

    private CompletableFuture<ShardTimeseriesStats> statistics(long[] tabletIds, long inactiveTime) {
        var index = new AtomicInteger();
        var completed = new AtomicInteger();
        var result = new AtomicReference<>(new ShardTimeseriesStats(0, 0));
        AsyncActorBody body = () -> {
            if (index.get() >= tabletIds.length) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            return statistics(tabletIds[index.getAndIncrement()], inactiveTime)
                    .thenAccept(stats -> {
                        result.updateAndGet(prev -> prev.combine(stats));
                        long progress = Math.round(((double) completed.incrementAndGet() / (double) tabletIds.length) * 100);
                        System.out.println("Progress: " + progress + "%");
                    });
        };

        var runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 50);
        return runner.start().thenApply(ignore -> result.get());
    }

    private CompletableFuture<ShardTimeseriesStats> statistics(long tabletId, long inactiveTime) {
        return latestTsByLocalId(tabletId)
                .thenApply(latestTsByLocalId -> {
                    long active = 0;
                    var it = latestTsByLocalId.long2LongEntrySet().fastIterator();
                    while (it.hasNext()) {
                        var entry = it.next();
                        if (entry.getLongValue() >= inactiveTime) {
                            active++;
                        }
                    }

                    var result = new ShardTimeseriesStats(active, latestTsByLocalId.size());

                    System.out.println("Done " + tabletId + " "
                            + DataSize.shortString(result.active)
                            + " / "
                            + DataSize.shortString(result.count) + " = " + result.activePercent() + "%");
                    return result;
                });
    }

    private CompletableFuture<Long2LongOpenHashMap> latestTsByLocalId(long tabletId) {
        return retry(() -> {
            return listIndexes(tabletId)
                    .thenCompose(listIndex -> latestTsByLocalId(tabletId, listIndex));
        });
    }

    private CompletableFuture<List<IndexAddressWithFileCount>> listIndexes(long tabletId) {
        return retry(() -> kvClient.readRangeNames(tabletId, 0, StockpileKvNames.currentFilesRange(), expiredAt()))
                .thenApply(response -> response.stream()
                        .map(file -> FileNameParsed.parseCurrent(file.getName()))
                        .filter(file -> file instanceof IndexFile)
                        .map(IndexFile.class::cast)
                        .collect(toList()))
                .thenApply(indices -> indices.stream()
                        .collect(collectingAndThen(toList(), IndexAddressWithFileCount::fold)));
    }

    private CompletableFuture<Long2LongOpenHashMap> latestTsByLocalId(long tabletId, List<IndexAddressWithFileCount> indexList) {
       var future = CompletableFuture.completedFuture(new Long2LongOpenHashMap());
       for (var index : indexList) {
           future = future.thenCompose(result -> {
               return latestTsByLocalId(tabletId, index)
                       .thenApply(latestPointsByIndex -> {
                           var it = latestPointsByIndex.long2LongEntrySet().fastIterator();
                           while (it.hasNext()) {
                               var entry = it.next();
                               if (result.get(entry.getLongKey()) < entry.getLongValue()) {
                                   result.put(entry.getLongKey(), entry.getLongValue());
                               }
                           }

                           return result;
                       });
           });
       }
       return future;
    }

    private CompletableFuture<Long2LongOpenHashMap> latestTsByLocalId(long tabletId, IndexAddressWithFileCount index) {
        return loadIndex(tabletId, index)
                .thenApply(content -> {
                    Long2LongOpenHashMap localIdToLatestTs = new Long2LongOpenHashMap();
                    for (var chunk : content.getChunks()) {
                        for (int metricIdx = 0; metricIdx < chunk.metricCount(); metricIdx++) {
                            long localId = chunk.getLocalId(metricIdx);
                            long latestTs = chunk.getLastTssMillis(metricIdx);
                            localIdToLatestTs.put(localId, latestTs);
                        }
                    }

                    return localIdToLatestTs;
                });
    }

    private CompletableFuture<SnapshotIndexContent> loadIndex(long tabletId, IndexAddressWithFileCount index) {
        CompletableFuture<ByteString> result = CompletableFuture.completedFuture(ByteString.EMPTY);
        for (var file : index.indexFileNames()) {
            result = result.thenCompose(prev -> retry(() -> kvClient.readData(tabletId, 0, file.reconstructCurrent(), expiredAt(), MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND))
                    .thenApply(bytes -> prev.concat(ByteStringsStockpile.unsafeWrap(bytes.orElseThrow()))));
        }

        return result.thenApply(bytes -> SnapshotIndexContentSerializer.S.deserializeToEof(new StockpileDeserializer(bytes)));
    }

    private static <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return RetryCompletableFuture.runWithRetries(supplier, RetryConfig.DEFAULT.withNumRetries(10));
    }

    private static long expiredAt() {
        return System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(5);
    }

    @Override
    public void close() {
        kvClient.close();
    }

    private record ShardTimeseriesStats(long active, long count) {
        public ShardTimeseriesStats combine(ShardTimeseriesStats other) {
            return new ShardTimeseriesStats(active + other.active, count + other.count);
        }

        public double activePercent() {
            return Math.round(((double) active / (double) count) * 100);
        }
    }
}
