package ru.yandex.solomon.experiments.gordiychuk;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.net.HostAndPort;
import com.yandex.ydb.table.TableClient;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.archive.header.DeleteBeforeField;
import ru.yandex.solomon.codec.archive.header.MetricHeader;
import ru.yandex.solomon.codec.bits.ReadOnlyHeapBitBuf;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardsDao;
import ru.yandex.solomon.experiments.gordiychuk.recovery.MetricsPushScheduler;
import ru.yandex.solomon.experiments.gordiychuk.recovery.metabase.MetabaseShardMetricsReader;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.cfg.SolomonPorts;
import ru.yandex.solomon.tool.stockpile.Metric;
import ru.yandex.solomon.tool.stockpile.StockpileShardWriters;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.stockpile.api.EDecimPolicy;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileActualizeMetricsHeader {
    private static final int MAX_SHARD_INFLIGHT = Runtime.getRuntime().availableProcessors();

    public static void main(String[] args) {
        MetricsPushScheduler.schedulePush();
        var cluster = SolomonCluster.valueOf(args[0]);
        var shardsDao= createShardsDao();
        var ydb = YdbHelper.createYdbClient(cluster);

        var shards = shardsDao.findAll().join();
        var writers = new StockpileShardWriters(stockpileClient(cluster), ForkJoinPool.commonPool());

        var notReadyShards = MetricRegistry.root().gaugeInt64("headers.update.not.finished.shards");
        var metrics = MetricRegistry.root().counter("processed.metrics");
        notReadyShards.set(shards.size());
        AtomicInteger current = new AtomicInteger();
        AsyncActorBody body = () -> {
            int index = current.getAndIncrement();
            if (index >= shards.size()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var shard = shards.get(index);

            double progress = index * 100. / shards.size();
            System.out.println("Start process: " + shard.getId());
            System.out.println("Process progress: " + String.format("%.2f%%", progress));
            MetabaseShardMetricsReader
                reader = new MetabaseShardMetricsReader(cluster.kikimrRootPath(), ydb, shard.getId(), 0);

            EProjectId projectId = shard.getProjectId().startsWith("yasm_")
                ? EProjectId.GOLOVAN
                : EProjectId.SOLOMON;
            EDecimPolicy decimPolicy = projectId == EProjectId.GOLOVAN
                ? EDecimPolicy.POLICY_5_MIN_AFTER_8_DAYS
                : EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS;

            var header = new MetricHeader(
                DeleteBeforeField.KEEP,
                projectId.getNumber(),
                shard.getNumId(),
                (short) decimPolicy.getNumber(),
                MetricType.METRIC_TYPE_UNSPECIFIED);

            var archive = new MetricArchiveImmutable(
                    header,
                    StockpileFormat.CURRENT,
                    0,
                    ReadOnlyHeapBitBuf.EMPTY,
                    0
                );

            return reader.run(record -> {
                writers.write(new Metric(record.shardId, record.localId, archive));
                metrics.inc();
            }).whenComplete((ignore, e) -> {
                notReadyShards.add(-1);
            });
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), MAX_SHARD_INFLIGHT);
        runner.start().join();
        writers.complete();
        writers.doneFuture().join();
        System.out.println("Success complete");
        System.exit(0);
    }

    private static StockpileClient stockpileClient(SolomonCluster cluster) {
        return StockpileClients.create(
            cluster.hosts().stream()
                .map(host -> HostAndPort.fromParts(host, SolomonPorts.STOCKPILE_GRPC))
                .collect(Collectors.toList()),
            StockpileClientOptions.newBuilder(
                    DefaultClientOptions.newBuilder()
                        .setRequestTimeOut(5, TimeUnit.MINUTES)
                        .setRpcExecutor(ForkJoinPool.commonPool())
                        .setResponseHandlerExecutorService(ForkJoinPool.commonPool())
                        .setClientId("syncHeaders")
                        .setMaxOutboundMessageSizeInBytes((int) DataSize.fromGigaBytes(100).toBytes())
                        .setMaxInboundMessageSizeInBytes((int) DataSize.fromMegaBytes(256).toBytes())
                )
                .setExpireClusterMetadata(30, TimeUnit.SECONDS)
                .setRetryStopStrategy(StopStrategies.stopAfterAttempt(1))
                .build());
    }

    private static ShardsDao createShardsDao() {
        SolomonCluster cluster = SolomonCluster.PROD_FRONT;
        TableClient tableClient = YdbHelper.createTableClient(cluster);
        return new YdbShardsDao(tableClient,  cluster.kikimrRootPath() + "/Config/V2/Shard", new ObjectMapper(), ForkJoinPool.commonPool());
    }
}
