package ru.yandex.solomon.experiments.gordiychuk;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import com.google.protobuf.ByteString;

import ru.yandex.kikimr.client.KikimrAsyncRetry;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClient.KvEntryStats;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.proto.MsgbusKv.TKeyValueRequest.EPriority;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.stockpile.kikimrKv.KvTabletsMapping;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.SnapshotAddress;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.ChunkFile;
import ru.yandex.stockpile.server.data.names.file.IndexFile;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class StockpileDebugCorruption {
    private static final Path FILE_CACHE_ROOT = Path.of("/home/gordiychuk/junk/corruption");

    public static void main(String[] args) throws IOException {
        SolomonCluster cluster = SolomonCluster.PROD_STORAGE_VLA;
        int shardId = 2920;
        var snapshot = new SnapshotAddress(SnapshotLevel.DAILY, 2936057720L);

        try (KikimrKvClient kvClient = KikimrHelper.createKvClient(cluster)) {
            ScheduledExecutorService scheduledExecutor = Executors.newSingleThreadScheduledExecutor();
            KvTabletsMapping kvMapping = new KvTabletsMapping(cluster.getSolomonVolumePath(), kvClient, scheduledExecutor, scheduledExecutor);
            kvMapping.waitForReady();


            long tabletId = kvMapping.getTabletId(shardId);
            var indexFileNames = listFiles(kvClient, tabletId, IndexFile.indexPrefixCurrentDotPf.format(snapshot)).join();
            printFiles("indexes:", indexFileNames);

            var chunkFileNames = listFiles(kvClient, tabletId, ChunkFile.chunkPrefixCurrentPf.format(snapshot)).join();
            printFiles("chunks:", chunkFileNames);

            System.out.println("load files...");
            var indexLoaded = loadFiles(kvClient, tabletId, indexFileNames).join();
            printLoadErrors(indexLoaded);

            var chunkLoaded = loadFiles(kvClient, tabletId, chunkFileNames).join();
            printLoadErrors(chunkLoaded);

            System.out.println("parse index...");
            var index = parseIndex(indexLoaded);

            System.out.println("check timeseries...");
            checkTimeseries(snapshot, index, chunkLoaded);

            System.err.println("done");
            System.exit(0);
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(-1);
        }
    }

    private static void printFiles(String type, List<KvEntryStats> files) {
        System.out.println(type);
        for (var file : files) {
            System.out.println(file);
        }
        System.out.println();
    }

    private static void printLoadErrors(List<LoadedFile> files) {
        for (var file : files) {
            if (file instanceof LoadedFileError f) {
                System.out.println(f.name + " error " + f.error.getMessage());
            }
        }
    }

    private static SnapshotIndexContent parseIndex(List<LoadedFile> files) {
        ByteString[] bytes = new ByteString[files.size()];
        for (int i = 0; i < files.size(); i++) {
            var file = files.get(i);
            if (file instanceof LoadedFileSuccess s) {
                bytes[i] = ByteStringsStockpile.unsafeWrap(s.content);
            } else {
                var e = (LoadedFileError) file;
                throw new IllegalArgumentException("index file not loaded from file: " + e.name, e.error);
            }
        }

        return SnapshotIndexContentSerializer.S.deserializeParts(bytes);
    }

    private static void checkTimeseries(SnapshotAddress address, SnapshotIndexContent index, List<LoadedFile> chunks) {
        int indexChunkNo = 0;
        int chunkIdx = 0;
        while (indexChunkNo < index.getChunksCount()) {
            var expectChunkName = StockpileKvNames.CURRENT_PREFIX + StockpileKvNames.chunkFileName(address.level(), address.txn(), indexChunkNo, FileNamePrefix.Current.instance);
            if (chunkIdx == chunks.size()) {
                chunkIdx++;
                System.out.println(expectChunkName + " file not exists");
                continue;
            }

            var chunkFileLoaded = chunks.get(chunkIdx);
            var chunkFileName = (ChunkFile) ChunkFile.parseCurrent(chunkFileLoaded.name());
            if (chunkFileName.chunkNo() > indexChunkNo) {
                indexChunkNo++;
                System.out.println(expectChunkName + " file not exists");
                continue;
            } else if (chunkFileName.chunkNo() < indexChunkNo) {
                chunkIdx++;
                continue;
            } else if (chunkFileLoaded instanceof LoadedFileError e) {
                chunkIdx++;
                indexChunkNo++;
                System.out.println(expectChunkName + " error " + e.error.getMessage());
                continue;
            }

            chunkIdx++;
            var chunkBytes = ((LoadedFileSuccess) chunkFileLoaded).content;
            var chunkIndex = index.getChunk(indexChunkNo++);
            for (int metricIdx = 0; metricIdx < chunkIndex.metricCount(); metricIdx++) {
                var localId = chunkIndex.getLocalId(metricIdx);
                var size = chunkIndex.getSize(metricIdx);
                var offset = chunkIndex.getOffset(metricIdx);

                try {
                    var archive = parseArchive(index.getFormat(), chunkBytes, offset, size);
                    iterateThrowArchive(archive, chunkIndex.getLastTssMillis(metricIdx));
                } catch (Throwable e) {
                    System.out.println(expectChunkName + " corrupted content,"
                            + " metricIdx: " + metricIdx
                            + " localId: " + localId
                            + " offset: " + offset
                            + " size: " + size
                            + " error: " + Throwables.getStackTraceAsString(e));
                    break;
                }
            }
        }
    }

    private static MetricArchiveImmutable parseArchive(StockpileFormat format, byte[] bytes, int offset, int length) {
        var deserializer = SnapshotIndexContentSerializer.dataDeserializerForVersionSealed(format);
        return deserializer.deserializeRange(bytes, offset, length);
    }

    private static void iterateThrowArchive(MetricArchiveImmutable archive, long expectLatestTsMillis) {
        var point = RecyclableAggrPoint.newInstance();
        long latestTsMillis = 0;
        try {
            int expectPoints = archive.getRecordCount();
            int iteratePoints = 0;
            var it = archive.iterator();
            while (it.next(point)) {
                iteratePoints++;
                if (latestTsMillis >= point.tsMillis) {
                    throw new IllegalStateException("prev ts " + latestTsMillis + " >= current ts " + point.tsMillis);
                }
                latestTsMillis = point.tsMillis;
            }

            if (expectPoints != iteratePoints) {
                throw new IllegalStateException("Points in header " + expectPoints + " points in real " + iteratePoints);
            }

            if (expectLatestTsMillis != latestTsMillis) {
                throw new IllegalStateException("expect latest ts " + expectLatestTsMillis + " !+ read latest ts " + latestTsMillis);
            }
        } finally {
            point.recycle();
        }
    }

    private static CompletableFuture<List<KvEntryStats>> listFiles(KikimrKvClient kvClient, long tabletId, String prefix) {
        return retry(() -> kvClient.readRangeNames(tabletId, 0, StringMicroUtils.asciiPrefixToRange(prefix), expiredAt()));
    }

    private static CompletableFuture<List<LoadedFile>> loadFiles(KikimrKvClient kvClient, long tabletId, List<KvEntryStats> files) {
        var future = CompletableFuture.<Void>completedFuture(null);
        var result = new ArrayList<LoadedFile>();
        for (var file : files) {
            future = future.thenCompose(ignore -> loadFile(kvClient, tabletId, file.getSize(), file.getName()))
                    .thenAccept(result::add);
        }
        return future.thenApply(ignore -> result);
    }

    private static CompletableFuture<LoadedFile> loadFile(KikimrKvClient kvClient, long tabletId, int expectSize, String name) {
        try {
            var cacheFile = FILE_CACHE_ROOT.resolve(name);
            if (Files.isReadable(cacheFile)) {
                var bytes = Files.readAllBytes(cacheFile);
                if (expectSize != bytes.length) {
                    throw new IllegalStateException("size different, expected " + expectSize + " but was " + bytes.length);
                }

                return CompletableFuture.completedFuture(new LoadedFileSuccess(name, bytes));
            }
        } catch (Throwable e) {
            // ignore cache
        }

        return kvClient.readData(tabletId, 0, name, expiredAt(), EPriority.BACKGROUND)
                .thenApply(opt -> {
                    var bytes = opt.orElseThrow();
                    if (expectSize != bytes.length) {
                        throw new IllegalStateException("size different, expected " + expectSize + " but was " + bytes.length);
                    }

                    try {
                        Files.write(FILE_CACHE_ROOT.resolve(name), bytes);
                    } catch (Throwable e) {
                        // ignore cache
                    }

                    return (LoadedFile) new LoadedFileSuccess(name, bytes);
                })
                .exceptionally(throwable -> new LoadedFileError(name, expectSize, Throwables.getRootCause(throwable)));
    }

    private static <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return KikimrAsyncRetry.withRetries(KikimrAsyncRetry.RetryConfig.maxRetries(10), supplier);
    }

    private static long expiredAt() {
        return System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(5);
    }

    sealed interface LoadedFile {
        String name();
    }

    private record LoadedFileSuccess(String name, byte[] content) implements LoadedFile {}
    private record LoadedFileError(String name, int size, Throwable error) implements LoadedFile {}
}
