package ru.yandex.solomon.experiments.gordiychuk;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Stream;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.client.KikimrAsyncRetry;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.cmdline.CmdArgsChief;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.codec.archive.serializer.MetricArchiveNakedSerializer;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.stockpile.kikimrKv.KvTabletsMapping;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.chunk.ChunkAddressGlobal;
import ru.yandex.stockpile.server.data.chunk.ChunkIndex;
import ru.yandex.stockpile.server.data.chunk.ChunkIterator;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.IndexAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.IndexFile;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileDumpMetrics {
    private StockpileSumpMetricsArgs args;
    private Path root;
    private final KikimrKvClient kvClient;
    private final KvTabletsMapping mapping;

    public StockpileDumpMetrics(StockpileSumpMetricsArgs args) {
        kvClient = KikimrHelper.createKvClient(args.cluster);
        this.root = Path.of(args.dir);
        this.args = args;
        mapping = new KvTabletsMapping(
            args.cluster.getSolomonVolumePath(),
            kvClient,
            ForkJoinPool.commonPool(),
            Executors.newSingleThreadScheduledExecutor()
        );
        mapping.waitForReady();
    }

    public CompletableFuture<?> dump() {
        return listIndexes()
            .thenCompose(this::loadIndex)
            .thenCompose(this::dumpIndex);
    }

    private CompletableFuture<?> dumpIndex(SnapshotIndex index) {
        var content = index.getContent();
        AtomicInteger current = new AtomicInteger();
        AsyncActorBody body = () -> {
            int chunkNo = current.getAndIncrement();
            if (chunkNo >= content.getChunksCount()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var address = new ChunkAddressGlobal(index.getLevel(), index.getTxn(), chunkNo);
            String progress = String.format("%.2f%%", chunkNo * 100. / content.getChunksCount());
            System.out.println("dump file: " + address + "  " + progress);
            return dumpChunk(address, content.getChunk(chunkNo));
        };

        return new AsyncActorRunner(body, ForkJoinPool.commonPool(), 5).start();
    }

    private CompletableFuture<?> dumpChunk(ChunkAddressGlobal address, ChunkIndex index) {
        long tableId = mapping.getTabletId(args.shardId);
        return retry(() -> kvClient.readDataLarge(tableId, 0, address.toString(), expiredAt(), MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND))
            .thenAccept(chunk -> {
                var iterator = new ChunkIterator(address, index, 0, chunk, StockpileFormat.CURRENT);
                while (iterator.hasNext()) {
                    MetricIdAndData data = iterator.next();
                    var projectId = data.archive().getOwnerProjectIdOrUnknown();
                    var type = data.archive().getType();
                    if (args.projectId.orElse(projectId) != projectId) {
                        continue;
                    }

                    if (args.type.orElse(type) != type) {
                        continue;
                    }

                    Path path = root.resolve(Integer.toString(args.shardId))
                        .resolve(projectId.name())
                        .resolve(type.name())
                        .resolve(Long.toString(data.localId()));

                    try {
                        ByteString bytes = MetricArchiveNakedSerializer.serializerForFormatSealed(StockpileFormat.CURRENT)
                            .serializeToByteString(data.archive());
                        path.getParent().toFile().mkdirs();
                        Files.write(path, bytes.toByteArray());
                    } catch (IOException e) {
                        throw new RuntimeException("failed at shard: " + args.shardId, e);
                    }
                }
            });
    }

    private CompletableFuture<SnapshotIndex> loadIndex(List<IndexAddressWithFileCount> address) {
        IndexAddressWithFileCount indexName = address.stream()
            .filter(i -> i.getLevel() == args.level.orElse(SnapshotLevel.DAILY))
            .min(Comparator.comparingLong(IndexAddressWithFileCount::getTxn))
            .orElse(null);

        if (indexName == null) {
            return CompletableFuture.failedFuture(new IllegalStateException("not found snapshot with level: " + args.level+" at shard " + args.shardId + " only " + address));
        }

        long tableId = mapping.getTabletId(args.shardId);
        return Stream.of(indexName.indexFileNames())
            .map(file -> retry(() -> kvClient.readData(tableId, 0, file.reconstructCurrent(), expiredAt(), MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND)))
            .collect(collectingAndThen(toList(), CompletableFutures::allOf))
            .thenApply(chunks -> {
                ByteString[] bytes = chunks.stream()
                    .map(buf -> ByteStringsStockpile.unsafeWrap(buf.get()))
                    .toArray(ByteString[]::new);

                SnapshotIndexContent indexContent = SnapshotIndexContentSerializer.S.deserializeParts(bytes);
                return new SnapshotIndex(indexName.getLevel(), indexName.getTxn(), indexContent);
            });
    }

    private CompletableFuture<List<IndexAddressWithFileCount>> listIndexes() {
        long tabletId = mapping.getTabletId(args.shardId);
        return retry(() -> kvClient.readRangeNames(tabletId, 0, StockpileKvNames.currentFilesRange(), expiredAt()))
            .thenApply(response -> response.stream()
                .map(file -> FileNameParsed.parseCurrent(file.getName()))
                .filter(file -> file instanceof IndexFile)
                .map(IndexFile.class::cast)
                .collect(toList()))
            .thenApply(indices -> {
                System.out.println("files: " + indices);
                return indices.stream()
                    .collect(collectingAndThen(toList(), IndexAddressWithFileCount::fold));
            });
    }

    private static <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return KikimrAsyncRetry.withRetries(KikimrAsyncRetry.RetryConfig.maxRetries(3), supplier);
    }

    private static long expiredAt() {
        return System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(5);
    }

    public static void main(String[] args) {
        StockpileSumpMetricsArgs parsedArgs = new StockpileSumpMetricsArgs();
        if (args.length == 0) {
            parsedArgs.shardId = 43;
            parsedArgs.level = Optional.of(SnapshotLevel.DAILY);
            parsedArgs.cluster = SolomonCluster.PROD_STORAGE_SAS;
            parsedArgs.dir = "/home/gordiychuk/junk/dump";
        } else {
            CmdArgsChief cmdArgsChief = new CmdArgsChief(parsedArgs, args);
        }

        try {
            new StockpileDumpMetrics(parsedArgs).dump().join();
        } catch (Throwable e) {
            e.printStackTrace();
        }

        System.exit(1);
    }
}
