package ru.yandex.solomon.experiments.gordiychuk;

import java.io.IOException;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import com.google.protobuf.ByteString;

import ru.yandex.kikimr.client.KikimrAsyncRetry;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.PropertyInitializer;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.stockpile.client.shard.StockpileMetricId;
import ru.yandex.stockpile.kikimrKv.ShardIdMapToLong;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContent;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.IndexAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.StockpileKvNames;
import ru.yandex.stockpile.server.data.names.file.IndexFile;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileHugeMetrics {
    private static KikimrKvClient kvClient;
    private static Writer writer;
    private static ShardIdMapToLong mapping;
    private static long bytesLimit;

    static {
        PropertyInitializer.init();
    }

    public static void main(String[] args) throws IOException {
        if (args.length < 2) {
            System.err.println("Usage: tool <size> <cluster_id> <file> <fromShardId>");
            System.exit(1);
        }

        SolomonCluster cluster = SolomonCluster.valueOf(args[1]);
        final Path path = Path.of(args[2]);
        if (path.getParent() != null) {
            Files.createDirectories(path.getParent());
        }
        bytesLimit = Long.parseLong(args[0]);
        writer = Files.newBufferedWriter(path, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        kvClient = KikimrHelper.createKvClient(cluster);
        mapping = retry(() -> kvClient.resolveKvTablets(cluster.getSolomonVolumePath()))
            .thenApply(ShardIdMapToLong::new)
            .join();

        int shardIdFrom = args.length == 4 ? Integer.valueOf(args[3]) : 0;
        var shards = mapping.shardIdStream()
            .filter(shardId -> shardId > shardIdFrom)
            .toArray();

        AtomicInteger cursor = new AtomicInteger();
        AsyncActorBody body = () -> {
            var pos = cursor.getAndIncrement();
            if (pos >= shards.length) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            String progress = String.format("%.2f%%", pos * 100. / shards.length);
            System.out.println("Progress: " + progress);
            return processShard(shards[pos]);
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), Runtime.getRuntime().availableProcessors() * 2);
        runner.start().join();
        writer.flush();
        writer.close();
        kvClient.close();
        System.exit(0);
    }

    private static CompletableFuture<Void> processShard(int shardId) {
        return retry(() -> listIndexes(shardId)
            .thenCompose(files -> processIndexes(shardId, files)))
            .whenComplete((s, e) -> {
                if (e != null) {
                    System.err.println("failed process shard " + shardId);
                    e.printStackTrace();
                    System.exit(1);
                }
            });
    }

    private static CompletableFuture<List<IndexAddressWithFileCount>> listIndexes(int shardId) {
        return retry(() -> kvClient.readRangeNames(mapping.get(shardId), 0, StockpileKvNames.currentFilesRange(), expiredAt()))
            .thenApply(response -> response.stream()
                .map(file -> FileNameParsed.parseCurrent(file.getName()))
                .filter(file -> file instanceof IndexFile)
                .map(IndexFile.class::cast)
                .collect(toList()))
            .thenApply(indices -> indices.stream()
                .collect(collectingAndThen(toList(), IndexAddressWithFileCount::fold)));
    }

    private static CompletableFuture<Void> processIndexes(int shardId, List<IndexAddressWithFileCount> address) {
        var tabletId = mapping.get(shardId);
        return address.stream()
            .filter(i -> i.getLevel() == SnapshotLevel.ETERNITY)
            .map(index -> Stream.of(index.indexFileNames())
                .map(file -> retry(() -> kvClient.readData(tabletId, 0, file.reconstructCurrent(), expiredAt(), MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND)))
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .thenAccept(chunks -> {
                    ByteString[] bytes = chunks.stream()
                        .map(buf -> ByteStringsStockpile.unsafeWrap(buf.get()))
                        .toArray(ByteString[]::new);

                    SnapshotIndexContent indexContent = SnapshotIndexContentSerializer.S.deserializeParts(bytes);
                    checkIndex(shardId, indexContent);
                }))
            .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOfVoid));
    }

    private static void checkIndex(int shardId, SnapshotIndexContent index) {
        Stream.of(index.getChunks())
            .parallel()
            .forEach(chunk -> IntStream.range(0, chunk.metricCount())
                .parallel()
                .forEach(pos -> {
                    var size = chunk.getSize(pos);
                    if (size >= bytesLimit) {
                        try {
                            writer.write(StockpileMetricId.toString(shardId, chunk.getLocalIdsSortedArray()[pos]) + " " + DataSize.prettyString(size) + "\n");
                        } catch (IOException e) {
                            throw new RuntimeException(e);
                        }
                    }
                }));
    }

    private static <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return KikimrAsyncRetry.withRetries(KikimrAsyncRetry.RetryConfig.maxRetries(10), supplier);
    }

    private static long expiredAt() {
        return System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(5);
    }
}
