package ru.yandex.solomon.experiments.gordiychuk;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;

import com.google.common.collect.Lists;
import com.google.common.net.HostAndPort;
import it.unimi.dsi.fastutil.objects.Object2DoubleMap;
import it.unimi.dsi.fastutil.objects.Object2DoubleOpenHashMap;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationAggregationSummary;
import ru.yandex.solomon.math.protobuf.OperationDropTimeSeries;
import ru.yandex.solomon.metrics.client.DcMetricsClient;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.FindResponse;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadManyResponse;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.timeseries.aggregation.DoubleSummary;
import ru.yandex.solomon.model.timeseries.aggregation.Int64Summary;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.cfg.SolomonPorts;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;

import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.metrics.client.ResponseValidationUtils.ensureMetabaseStatusValid;
import static ru.yandex.solomon.metrics.client.ResponseValidationUtils.ensureStockpileStatusValid;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileTopShardBySize implements AutoCloseable {
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(100)
            .withDelay(1_000)
            .withMaxDelay(60_000L)
            .withStats((timeSpentMillis, cause) -> {
                cause.printStackTrace();
            });

    private final MetricsClient client;

    public StockpileTopShardBySize(MetricsClient client) {
        this.client = client;
    }

    public static void main(String[] args) {
        var selectors = Selectors.parse("""
                project="solomon",
                cluster="stockpile_sas",
                service="stockpile_resources",
                kind="total",
                host="cluster",
                sensor="stockpile.host.bytes.count",
                ownerId!="total",
                projectId="GOLOVAN|SOLOMON",
                level="total"
                """);

        try (var task = new StockpileTopShardBySize(createMetricsClient())) {
            task.run(selectors).join();
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    public CompletableFuture<Void> run(Selectors selectors) {
        return find(selectors)
                .thenCompose(this::read)
                .thenAccept(result -> {
                    List<Record> records = new ArrayList<>();
                    for (var entry : result.object2DoubleEntrySet()) {
                        var numId = entry.getKey().getLabels().findByKey("ownerId").getValue();
                        records.add(new Record(numId, entry.getDoubleValue()));
                    }

                    records.sort(Comparator.comparingDouble(Record::value).reversed());
                    var top = records.subList(0, Math.min(records.size(), 100));
                    for (var record : top) {
                        System.out.println(record.numId + " " + Math.round(record.value) + " " + DataSize.prettyString(Math.round(record.value)));
                    }
                });
    }

    private static record Record(String numId, double value) {
    }

    private CompletableFuture<FindResponse> find(FindRequest request) {
        return RetryCompletableFuture.runWithRetries(() -> client.find(request)
                .thenApply(response -> {
                    ensureMetabaseStatusValid(response.getStatus());
                    return response;
                }), RETRY_CONFIG);
    }

    private CompletableFuture<ReadManyResponse> read(ReadManyRequest request) {
        return RetryCompletableFuture.runWithRetries(() -> client.readMany(request)
                .thenApply(response -> {
                    ensureStockpileStatusValid(response.getStatus());
                    return response;
                }), RETRY_CONFIG);
    }

    private CompletableFuture<List<MetricKey>> find(Selectors selectors) {
        int limit = 10000;
        List<MetricKey> result = new ArrayList<>();
        AsyncActorBody body = () -> {
            var req = FindRequest.newBuilder()
                    .setOffset(result.size())
                    .setLimit(limit)
                    .setSelectors(selectors)
                    .build();

            return find(req)
                    .thenApply(response -> {
                        result.addAll(response.getMetrics());
                        if (response.isTruncated() && result.size() < response.getTotalCount()) {
                            return null;
                        }

                        return AsyncActorBody.DONE_MARKER;
                    });
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 1);
        return runner.start().thenApply(ignore -> result);
    }

    private CompletableFuture<Object2DoubleMap<MetricKey>> read(List<MetricKey> keys) {
        var it = Lists.partition(keys, 1000).iterator();

        var result = new Object2DoubleOpenHashMap<MetricKey>(keys.size());
        var now = System.currentTimeMillis();
        AsyncActorBody body = () -> {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            return read(now, it.next())
                    .thenAccept(result::putAll)
                    .thenAccept(ignore -> {
                        System.out.println("read " + result.size());
                    });
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 1);
        return runner.start().thenApply(ignore -> result);
    }

    private CompletableFuture<Object2DoubleMap<MetricKey>> read(long now, List<MetricKey> part) {
        var req = ReadManyRequest.newBuilder()
                .addKeys(part)
                .setFromMillis(now - TimeUnit.MINUTES.toMillis(5L))
                .setToMillis(now)
                .addOperation(Operation.newBuilder()
                        .setSummary(OperationAggregationSummary.newBuilder()
                                .addAggregations(Aggregation.MAX)
                                .build())
                        .build())
                .addOperation(Operation.newBuilder()
                        .setDropTimeseries(OperationDropTimeSeries.newBuilder().build())
                        .build())
                .build();

        return read(req)
                .thenApply(response -> {
                    var result = new Object2DoubleOpenHashMap<MetricKey>(response.getMetrics().size());
                    for (var metric : response.getMetrics()) {
                        var key = metric.getKey();
                        if (key == null) {
                            continue;
                        }

                        var summary = metric.getSummary();
                        if (summary == null || summary.getCount() == 0) {
                            continue;
                        }

                        final double value;
                        if (summary instanceof DoubleSummary) {
                            value = ((DoubleSummary) summary).getMax();
                        } else if (summary instanceof Int64Summary) {
                            value = ((Int64Summary) summary).getMax();
                        } else {
                            continue;
                        }

                        result.addTo(metric.getKey(), value);
                    }

                    return result;
                });

    }

    private static MetricsClient createMetricsClient() {
        var metabase = createMetabaseClient(SolomonCluster.PROD_FETCHER_VLA);
        metabase.forceUpdateClusterMetaData().join();
        var stockpile = createStockpileClient(SolomonCluster.PROD_STOCKPILE_VLA);
        stockpile.forceUpdateClusterMetaData().join();
        return new DcMetricsClient("vla", metabase, stockpile);
    }

    private static MetabaseClient createMetabaseClient(SolomonCluster cluster) {
        List<HostAndPort> addresses = cluster.hosts()
                .stream()
                .map(s -> HostAndPort.fromParts(s, SolomonPorts.COREMON_GRPC))
                .collect(toList());

        var options = MetabaseClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                        .setRequestTimeOut(10, TimeUnit.MINUTES)
                        .setKeepAliveTimeout(10, TimeUnit.MINUTES)
                        .setIdleTimeOut(10, TimeUnit.HOURS))
                .setExpireClusterMetadata(30, TimeUnit.MINUTES)
                .setMetaDataRequestTimeOut(5, TimeUnit.MINUTES)
                .build();

        return MetabaseClients.create(addresses, options);
    }

    private static StockpileClient createStockpileClient(SolomonCluster cluster) {
        List<HostAndPort> addresses = cluster.addressesStockpileGrpc();

        var options = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                        .setRequestTimeOut(10, TimeUnit.MINUTES)
                        .setMaxInboundMessageSizeInBytes(500 << 20)) // 500 Mib
                .setExpireClusterMetadata(30, TimeUnit.SECONDS)
                .setRetryStopStrategy(StopStrategies.stopAfterAttempt(1000))
                .build();

        return StockpileClients.create(addresses, options);
    }

    @Override
    public void close() {
        client.close();
    }
}
