package ru.yandex.solomon.experiments.gordiychuk;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.google.common.net.HostAndPort;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2DoubleMap;
import it.unimi.dsi.fastutil.objects.Object2DoubleOpenHashMap;
import joptsimple.internal.Strings;
import org.HdrHistogram.Histogram;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationAggregationSummary;
import ru.yandex.solomon.math.protobuf.OperationDropTimeSeries;
import ru.yandex.solomon.metrics.client.DcMetricsClient;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.metrics.client.FindResponse;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadManyRequest;
import ru.yandex.solomon.metrics.client.ReadManyResponse;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.timeseries.aggregation.DoubleSummary;
import ru.yandex.solomon.model.timeseries.aggregation.Int64Summary;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.cfg.SolomonPorts;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;

import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.metrics.client.ResponseValidationUtils.ensureMetabaseStatusValid;
import static ru.yandex.solomon.metrics.client.ResponseValidationUtils.ensureStockpileStatusValid;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileWriteDistributionByService implements AutoCloseable {
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(100)
            .withDelay(1_000)
            .withMaxDelay(60_000L)
            .withStats((timeSpentMillis, cause) -> {
                cause.printStackTrace();
            });

    private final MetricsClient client;

    public StockpileWriteDistributionByService(MetricsClient client) {
        this.client = client;
    }

    public static void main(String[] args) {
        var selectors = Selectors.parse("""
                project="solomon_cloud",
                cluster="stockpile_vla",
                service="stockpile_resources",
                kind="total",
                host="cluster",
                producer="total",
                sensor="stockpile.write.records.rate",
                projectId="SOLOMON",
                ownerId!="total"
                """);

        try (var task = new StockpileWriteDistributionByService(createMetricsClient())) {
            task.run(selectors).join();
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(1);
        }

        System.exit(0);
    }

    public CompletableFuture<Void> run(Selectors selectors) {
        return find(selectors)
                .thenCompose(this::read)
                .thenAccept(result -> {
                    var numIdToService = numIdToService();
                    var serviceStats = new HashMap<String, Stat>();
                    for (var entry : result.object2DoubleEntrySet()) {
                        if (entry.getDoubleValue() <= 0.0) {
                            continue;
                        }

                        var numId = numId(entry.getKey().getLabels().findByKey("ownerId").getValue());
                        if (numId == 0) {
                            continue;
                        }

                        var service = numIdToService.get(numId);
                        if (Strings.isNullOrEmpty(service)) {
                            continue;
                        }

                        var stats = serviceStats.get(service);
                        if (stats == null) {
                            stats = new Stat(service);
                            serviceStats.put(service, stats);
                        }

                        stats.count++;
                        stats.sum += Math.round(entry.getDoubleValue());
                        stats.histogram.recordValue(Math.round(entry.getDoubleValue()));
                    }

                    var list = serviceStats.values().stream()
                            .filter(stat -> stat.count >= 100)
                            .sorted(Comparator.<Stat>comparingInt(o -> o.count).reversed())
                            .map(Stat::toRecord)
                            .collect(Collectors.toList());

                    printTable(list);
                });
    }

    private void printTable(List<TableRecord> records) {
        var size = records.stream()
                .mapToInt(value -> value.service.length()).max()
                .orElse(10) + 3;

        System.out.printf("%-" + size + "s", "service");
        System.out.printf("%-8s", "shards");
        System.out.printf("%-8s", "sum");
        System.out.printf("%-8s", "p50");
        System.out.printf("%-8s", "p90");
        System.out.printf("%-8s", "p99");
        System.out.printf("%-8s", "p99.9");
        System.out.println();

        for (var record : records) {
            System.out.printf("%-" + size + "s", record.service);
            System.out.printf("%-8s", record.count);
            System.out.printf("%-8s", record.sum);
            System.out.printf("%-8s", record.p50);
            System.out.printf("%-8s", record.p90);
            System.out.printf("%-8s", record.p99);
            System.out.printf("%-8s", record.p99_9);
            System.out.println();
        }
    }

    private int numId(String numId) {
        try {
            return Integer.parseUnsignedInt(numId);
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    private Int2ObjectMap<String> numIdToService() {
        try (var stream = Files.lines(Path.of("/home/gordiychuk/junk/shards.json"))) {
            Int2ObjectMap<String> result = new Int2ObjectOpenHashMap<>();
            ObjectMapper mapper = new ObjectMapper();
            var it = stream.iterator();
            while (it.hasNext()) {
                var line = it.next();
                var shard = mapper.readValue(line, Shard.class);
                result.put(shard.numId, shard.service);
            }

            return result;
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    private CompletableFuture<FindResponse> find(FindRequest request) {
        return RetryCompletableFuture.runWithRetries(() -> client.find(request)
                .thenApply(response -> {
                    ensureMetabaseStatusValid(response.getStatus());
                    return response;
                }), RETRY_CONFIG);
    }

    private CompletableFuture<ReadManyResponse> read(ReadManyRequest request) {
        return RetryCompletableFuture.runWithRetries(() -> client.readMany(request)
                .thenApply(response -> {
                    ensureStockpileStatusValid(response.getStatus());
                    return response;
                }), RETRY_CONFIG);
    }

    private CompletableFuture<List<MetricKey>> find(Selectors selectors) {
        int limit = 10000;
        List<MetricKey> result = new ArrayList<>();
        AsyncActorBody body = () -> {
            var req = FindRequest.newBuilder()
                    .setOffset(result.size())
                    .setLimit(limit)
                    .setSelectors(selectors)
                    .build();

            return find(req)
                    .thenApply(response -> {
                        result.addAll(response.getMetrics());
                        if (response.isTruncated() && result.size() < response.getTotalCount()) {
                            return null;
                        }

                        return AsyncActorBody.DONE_MARKER;
                    });
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 1);
        return runner.start().thenApply(ignore -> result);
    }

    private CompletableFuture<Object2DoubleMap<MetricKey>> read(List<MetricKey> keys) {
        var it = Lists.partition(keys, 10000).iterator();

        var result = new Object2DoubleOpenHashMap<MetricKey>(keys.size());
        var now = System.currentTimeMillis();
        AsyncActorBody body = () -> {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            return read(now, it.next())
                    .thenAccept(result::putAll)
                    .thenAccept(ignore -> {
                        System.out.println("read " + result.size());
                    });
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), 1);
        return runner.start().thenApply(ignore -> result);
    }

    private CompletableFuture<Object2DoubleMap<MetricKey>> read(long now, List<MetricKey> part) {
        var aggr = Aggregation.AVG;
        var req = ReadManyRequest.newBuilder()
                .addKeys(part)
                .setFromMillis(now - TimeUnit.MINUTES.toMillis(10L))
                .setToMillis(now  - TimeUnit.MINUTES.toMillis(5L))
                .addOperation(Operation.newBuilder()
                        .setSummary(OperationAggregationSummary.newBuilder()
                                .addAggregations(aggr)
                                .build())
                        .build())
                .addOperation(Operation.newBuilder()
                        .setDropTimeseries(OperationDropTimeSeries.newBuilder().build())
                        .build())
                .build();

        return read(req)
                .thenApply(response -> {
                    var result = new Object2DoubleOpenHashMap<MetricKey>(response.getMetrics().size());
                    for (var metric : response.getMetrics()) {
                        var key = metric.getKey();
                        if (key == null) {
                            continue;
                        }

                        var summary = metric.getSummary();
                        if (summary == null || summary.getCount() == 0 || !summary.has(aggr)) {
                            continue;
                        }

                        final double value;
                        if (summary instanceof DoubleSummary) {
                            value = ((DoubleSummary) summary).getAvg();
                        } else if (summary instanceof Int64Summary) {
                            value = ((Int64Summary) summary).getAvg();
                        } else {
                            continue;
                        }

                        result.addTo(metric.getKey(), value);
                    }

                    return result;
                });

    }

    private static MetricsClient createMetricsClient() {
        var metabase = createMetabaseClient(SolomonCluster.PROD_FETCHER_VLA);
        metabase.forceUpdateClusterMetaData().join();
        System.out.println("init metabase");
        var stockpile = createStockpileClient(SolomonCluster.PROD_STOCKPILE_VLA);
        stockpile.forceUpdateClusterMetaData().join();
        System.out.println("init stockpile");
        return new DcMetricsClient("vla", metabase, stockpile);
    }

    private static MetabaseClient createMetabaseClient(SolomonCluster cluster) {
        List<HostAndPort> addresses = cluster.hosts()
                .stream()
                .map(s -> HostAndPort.fromParts(s, SolomonPorts.COREMON_GRPC))
                .collect(toList());

        var options = MetabaseClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                        .setRequestTimeOut(10, TimeUnit.MINUTES)
                        .setKeepAliveTimeout(10, TimeUnit.MINUTES)
                        .setIdleTimeOut(10, TimeUnit.HOURS))
                .setExpireClusterMetadata(30, TimeUnit.MINUTES)
                .setMetaDataRequestTimeOut(5, TimeUnit.MINUTES)
                .build();

        return MetabaseClients.create(addresses, options);
    }

    private static StockpileClient createStockpileClient(SolomonCluster cluster) {
        List<HostAndPort> addresses = cluster.addressesStockpileGrpc();

        var options = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setRequestTimeOut(10, TimeUnit.MINUTES)
                    .setMaxInboundMessageSizeInBytes(500 << 20)) // 500 Mib
                .setExpireClusterMetadata(30, TimeUnit.SECONDS)
                .setRetryStopStrategy(StopStrategies.stopAfterAttempt(1000))
                .build();

        return StockpileClients.create(addresses, options);
    }

    @Override
    public void close() {
        client.close();
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class Shard {
        @JsonProperty("numId")
        private int numId;
        @JsonProperty("serviceName")
        private String service;
    }

    private static class Stat {
        private final String service;
        private Histogram histogram = new Histogram(2);
        private int count;
        private long sum;

        public Stat(String service) {
            this.service = service;
        }

        public TableRecord toRecord() {
            return new TableRecord(service,
                    DataSize.shortString(count),
                    DataSize.shortString(histogram.getValueAtPercentile(50.0)),
                    DataSize.shortString(histogram.getValueAtPercentile(90.0)),
                    DataSize.shortString(histogram.getValueAtPercentile(99.0)),
                    DataSize.shortString(histogram.getValueAtPercentile(99.9)),
                    DataSize.shortString(sum));
        }
    }

    private static record TableRecord(String service, String count, String p50, String p90, String p99, String p99_9, String sum) {
    }
}
