package ru.yandex.solomon.experiments.gordiychuk;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.LongStream;
import java.util.stream.Stream;

import ru.yandex.discovery.DiscoveryServices;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.StringMicroUtils;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonPorts;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class TruncateDumperKv {
    private static final long BYTES_LIMIT = 5 << 20; // 5 Mib

    public static void main(String[] args) {
        var hosts = DiscoveryServices.resolve("conductor_group://solomon_prod_sts1_ydb_myt:" + SolomonPorts.KIKIMR_GRPC);
//        var cluster = SolomonCluster.TEST_STS;
        KikimrKvClient kvClient = KikimrHelper.createKvClient(hosts);

        List<NameRange> delete = new ArrayList<>();
        delete.add(NameRange.all());
        delete.add(StringMicroUtils.asciiPrefixToRange("c.d.l."));
        delete.add(StringMicroUtils.asciiPrefixToRange("c.m.s."));

        kvClient.resolveKvTablets("/Solomon/ShortTermStorage/KV")
            .thenCompose(tabletIds ->
                LongStream.of(tabletIds)
                    .mapToObj(tabletId -> rmRange(kvClient, tabletId, delete)
                        .thenCompose(unit -> kvClient.incrementGeneration(tabletId, expiredAt()))
                        .handle((ignore, e) -> {
                            if (e != null) {
                                System.err.println("tablet failed: "+ tabletId);
                                e.printStackTrace();
                            } else {
                                System.out.println(tabletId + " done");
                            }
                            return null;
                        }))
                    .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOfVoid))
            )
            .join();
        System.exit(0);
    }

    private static CompletableFuture<Void> rmRange(KikimrKvClient kvClient, long tabletId, List<NameRange> delete) {
//        return kvClient.deleteRanges(tabletId, 0, delete, expiredAt());
        var future = CompletableFuture.<Void>completedFuture(null);
        for (var range : delete) {
            future = future.thenCompose(ignore -> rmFiles(kvClient, tabletId, range));
        }
        return future;
    }

    private static CompletableFuture<Void> rmFiles(KikimrKvClient kvClient, long tabletId, NameRange range) {
        return kvClient.readRange(tabletId, 0, range, false, BYTES_LIMIT, expiredAt())
                .thenCompose(result -> {
                    var names = Stream.of(result.getEntries())
                            .map(KikimrKvClient.KvEntryWithStats::getName)
                            .collect(Collectors.toList());
                    return kvClient.deleteFiles(tabletId, 0, names, expiredAt())
                            .thenCompose(ignore -> {
                                if (!result.isOverrun()) {
                                    return CompletableFuture.completedFuture(null);
                                }

                                System.out.println(tabletId + " rm files " + names.size());
                                return rmFiles(kvClient, tabletId, range);
                            });
                });
    }

    private static long expiredAt() {
        return System.currentTimeMillis() + TimeUnit.MINUTES.toMillis(10L);
    }
}
