package ru.yandex.solomon.experiments.gordiychuk;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Random;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.solomon.codec.archive.MetricArchiveMutable;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.point.column.TsRandomData;
import ru.yandex.solomon.model.point.column.ValueRandomData;

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, time = 1, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 2, time = 1, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.Throughput)
@OutputTimeUnit(TimeUnit.SECONDS)
public class TxWriteBenchmark {
    @Param({"1", "10", "100"})
    public int requestCount;

    private List<Long2ObjectOpenHashMap<MetricArchiveMutable>> requests = new ArrayList<>();

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(TxWriteBenchmark.class.getName())
                .detectJvmArgs()
//                .addProfiler(AsyncProfiler.class)
                .build();

        new Runner(opt).run();
    }

    @Setup
    public void setUp() {
        requests = IntStream.range(0, requestCount)
                .parallel()
                .mapToObj(rIndex -> {
                    AggrPoint point = new AggrPoint(StockpileColumn.TS.mask() | StockpileColumn.VALUE.mask());
                    Random random = ThreadLocalRandom.current();
                    Long2ObjectOpenHashMap<MetricArchiveMutable> r = new Long2ObjectOpenHashMap<>();
                    for (int sIndex = 0; sIndex < 4000; sIndex++) {
                        long localId = ThreadLocalRandom.current().nextLong(100000, 100500);
//                        long localId = StockpileLocalId.random(random);
                        MetricArchiveMutable archive = new MetricArchiveMutable();
                        int points = ThreadLocalRandom.current().nextInt(1, 3);
                        for (int pIndex = 0; pIndex < points; pIndex++) {
                            point.tsMillis = TsRandomData.randomTs(random);
                            point.valueNum = ValueRandomData.randomNum(random);
                            archive.addRecord(point);
                        }
                        r.put(localId, archive);
                    }
                    return r;
                })
                .collect(Collectors.toList());
    }

    @Benchmark
    public Object syncCombine() {
        Long2ObjectOpenHashMap<MetricArchiveMutable> archiveByLocalId = new Long2ObjectOpenHashMap<>(5000);
        for (Long2ObjectOpenHashMap<MetricArchiveMutable> request : requests) {
            for (Long2ObjectMap.Entry<MetricArchiveMutable> entry : request.long2ObjectEntrySet()) {
                MetricArchiveMutable archive = archiveByLocalId.get(entry.getLongKey());
                if (archive != null) {
                    archive.updateWithNoSortMerge(entry.getValue());
                } else {
                    archiveByLocalId.put(entry.getLongKey(), entry.getValue());
                }
            }
        }

        return archiveByLocalId;
    }

    @Benchmark
    public Object asyncCombine() {
        Long2ObjectOpenHashMap<MetricArchiveMutable> archiveByLocalId = new Long2ObjectOpenHashMap<>(5000);

        for (Long2ObjectOpenHashMap<MetricArchiveMutable> request : requests) {
            request.long2ObjectEntrySet()
                    .parallelStream()
                    .map(entry -> {
                        MetricArchiveMutable archive = archiveByLocalId.get(entry.getLongKey());
                        if (archive != null) {
                            archive.updateWithNoSortMerge(entry.getValue());
                            return null;
                        }

                        return entry;
                    })
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList())
                    .forEach(entry -> archiveByLocalId.put(entry.getLongKey(), entry.getValue()));
        }

        return archiveByLocalId;
    }

    @Benchmark
    public Object asyncCombine2() {
        Long2ObjectOpenHashMap<MetricArchiveMutable> archiveByLocalId = new Long2ObjectOpenHashMap<>(5000);

        for (Long2ObjectOpenHashMap<MetricArchiveMutable> request : requests) {
            request.long2ObjectEntrySet()
                    .parallelStream()
                    .map(entry -> {
                        MetricArchiveMutable archive = archiveByLocalId.get(entry.getLongKey());
                        if (archive != null) {
                            archive.updateWithNoSortMerge(entry.getValue());
                            return null;
                        }

                        return entry;
                    })
                    .filter(Objects::nonNull)
                    .sequential()
                    .forEach(entry -> archiveByLocalId.put(entry.getLongKey(), entry.getValue()));
        }

        return archiveByLocalId;
    }
}
