package ru.yandex.solomon.experiments.gordiychuk;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.net.HostAndPort;
import com.yandex.ydb.table.TableClient;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardsDao;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.metrics.client.DcMetricsClient;
import ru.yandex.solomon.metrics.client.MetabaseClientException;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.ReadRequest;
import ru.yandex.solomon.metrics.client.ResolveOneRequest;
import ru.yandex.solomon.metrics.client.StockpileClientException;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.cfg.SolomonPorts;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class UpYasmQuota {
    private static Labels COMMON_LABELS = Labels.builder(5)
        .add("project", "solomon")
        .add("cluster", "production")
        .add("service", "coremon")
        .add("host", "Vla")
        .add("sensor", "engine.fileSensors")
        .build();

    private static ShardsDao dao;
    private static MetricsClient metrics;

    public static void main(String[] args) throws IOException {
        dao = createShardsDao();
        metrics = createMetricsClient();

        var shards = dao.findAll().join()
            .stream()
            .filter(shard -> shard.getProjectId().startsWith("yasm_"))
            .filter(shard -> !"yasm_yappy".equals(shard.getProjectId()))
            .filter(shard -> !"yasm_marketdesktop".equals(shard.getProjectId()))
            .collect(Collectors.toList());

        var writer = Files.newBufferedWriter(Path.of("/home/gordiychuk/junk/quota.txt"), StandardOpenOption.CREATE, StandardOpenOption.APPEND);
        writer.write("projectId; shardId; usage; max; update\n");
        for (var shard : shards) {
            var usage = quotaUsage(shard).join();

            double max = shard.getMaxFileMetrics();
            long percent = Math.round(100 * (usage/max));

            if (percent < 79) {
                continue;
            }

            long newQuota = Math.round(max + (max * 0.5));
            System.out.println("shard: " + shard.getId() + " " + percent);
            writer.write(shard.getProjectId() + ";");
            writer.write(shard.getId() + ";");
            writer.write(Math.round(usage) + ";");
            writer.write(shard.getMaxFileMetrics() + ";");
            writer.write(newQuota + "\n");

            var update = shard.toBuilder()
                .setMaxFileMetrics(Math.toIntExact(newQuota))
                .build();
            dao.partialUpdate(update, true).join();
        }

        writer.flush();
        writer.close();
        System.exit(0);
    }

    private static CompletableFuture<Double> quotaUsage(Shard shard) {
        return metrics.resolveOne(ResolveOneRequest.newBuilder()
            .setLabels(COMMON_LABELS.addAll(Labels.of("shardId", shard.getId(), "projectId", shard.getProjectId())))
            .build())
            .thenApply(response -> {
                if (!response.isOk()) {
                    throw new MetabaseClientException(response.getStatus());
                }

                return response.getMetric();
            })
            .thenCompose(key -> metrics.read(ReadRequest.newBuilder()
                .setKey(key)
                .setFrom(Instant.now().minus(5, ChronoUnit.MINUTES))
                .setTo(Instant.now())
                .build()))
            .thenApply(response -> {
                if (!response.isOk()) {
                    throw new StockpileClientException(response.getStatus());
                }

                var graph = AggrGraphDataArrayList.of(response.getSource());
                return graph.getValueDivided(graph.length() - 1);
            });
    }

    private static MetricsClient createMetricsClient() {
        var metabase = createMetabaseClient(SolomonCluster.PROD_FETCHER_VLA);
        var stockpile = createStockpileClient(SolomonCluster.PROD_STORAGE_VLA);
        return new DcMetricsClient("vla", metabase, stockpile);
    }

    private static MetabaseClient createMetabaseClient(SolomonCluster cluster) {
        List<HostAndPort> addresses = cluster.hosts()
            .stream()
            .map(s -> HostAndPort.fromParts(s, SolomonPorts.COREMON_GRPC))
            .collect(toList());

        var options = MetabaseClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setRequestTimeOut(10, TimeUnit.MINUTES)
                    .setKeepAliveTimeout(10, TimeUnit.MINUTES)
                    .setIdleTimeOut(10, TimeUnit.HOURS))
            .setExpireClusterMetadata(30, TimeUnit.MINUTES)
            .setMetaDataRequestTimeOut(5, TimeUnit.MINUTES)
            .build();

        return MetabaseClients.create(addresses, options);
    }

    private static StockpileClient createStockpileClient(SolomonCluster cluster) {
        List<HostAndPort> addresses = cluster.addressesStockpileGrpc();

        var options = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setRequestTimeOut(10, TimeUnit.MINUTES)
                    .setMaxInboundMessageSizeInBytes(500 << 20)) // 500 Mib
            .setExpireClusterMetadata(30, TimeUnit.SECONDS)
            .setRetryStopStrategy(StopStrategies.stopAfterAttempt(1000))
            .build();

        return StockpileClients.create(addresses, options);
    }

    private static ShardsDao createShardsDao() {
        SolomonCluster cluster = SolomonCluster.PROD_FRONT;
        TableClient tableClient = YdbHelper.createTableClient(cluster);
        return new YdbShardsDao(tableClient,  cluster.kikimrRootPath() + "/Config/V2/Shard", new ObjectMapper(), ForkJoinPool.commonPool());
    }

    private static class Usage {
        final Shard shard;
        final double usage;
        final double limit;

        public Usage(Shard shard, double usage, double limit) {
            this.shard = shard;
            this.usage = usage;
            this.limit = limit;
        }
    }
}
