package ru.yandex.solomon.experiments.gordiychuk;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.misc.lang.DefaultToString;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.ClusterHostListConf;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.gateway.api.v2.dto.ClusterDto;
import ru.yandex.solomon.util.jackson.SalmonJackson;

import static ru.yandex.solomon.core.db.model.ShardSettings.AggregationSettings.EMPTY;


/**
 * @author Vladimir Gordiychuk
 */
public class YasmSyncTsdbHosts {
    private static final List<String> NANNY_SERVICES = List.of(
            "production_yasm_tsdb_kernel_test_sas",
            "production_yasm_tsdb_sas",
            "production_yasm_tsdb_misc_sas",
            "production_yasm_tsdb_misc_man",
            "production_yasm_tsdb_misc_vla",
            "production_yasm_tsdb_misc_msk_new",
            "production_yasm_tsdb_man",
            "production_yasm_tsdb_vla",
            "production_yasm_tsdb_iva_new",
            "production_yasm_tsdb_myt_new");

    public static void main(String[] args) throws IOException, InterruptedException {
        var http = HttpClient.newHttpClient();
        var mapper = new ObjectMapper();
        var replica_0 = new ArrayList<ClusterHostListConf>();
        var replica_1 = new ArrayList<ClusterHostListConf>();

        for (String nanny_service : NANNY_SERVICES) {
            var uri = URI.create("http://nanny.yandex-team.ru/v2/services/" + nanny_service + "/current_state/instances/");
            var resp = http.send(HttpRequest.newBuilder()
                    .GET()
                    .uri(uri)
                    .build(), HttpResponse.BodyHandlers.ofString());

            if (resp.statusCode() != 200) {
                throw new IllegalStateException("Not 200 " + resp);
            }

            var instances = mapper.readValue(resp.body(), Response.class).result;
            for (var instance : instances) {
                List<ClusterHostListConf> target;
                if (instance.tags.contains("itag_replica_0")) {
                    target = replica_0;
                } else {
                    target = replica_1;
                }

                target.add(ClusterHostListConf.of(instance.hostname, null, null, new String[]{"nanny="+nanny_service}));
            }
        }

        System.out.println(prepareJson(0, replica_0));
        System.out.println(prepareJson(1, replica_1));
    }

    private static String prepareJson(int replica, List<ClusterHostListConf> hosts) throws JsonProcessingException {
        var cluster = Cluster.newBuilder()
                .setId("yasm_prod_tsdb_replica_" + replica)
                .setName("production_replica_" + replica)
                .setProjectId("yasm")
                .setHosts(hosts)
                .setCreatedAt(Instant.now())
                .setUpdatedAt(Instant.now())
                .setCreatedBy("gordiychuk")
                .setUpdatedBy("gordiychuk")
                .setShardSettings(ShardSettings.of(
                        ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        EMPTY,
                        0
                ))
                .build();

        var dto = ClusterDto.fromModel(cluster);
        var mapper = SalmonJackson.getObjectMapper();
        return mapper.writeValueAsString(dto);
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    @JsonIgnoreProperties(ignoreUnknown=true)
    public static class Response extends DefaultToString {
        @JsonProperty("result")
        List<Instance> result;
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    @JsonIgnoreProperties(ignoreUnknown=true)
    public static class Instance extends DefaultToString {
        @JsonProperty("hostname")
        public String hostname;
        @JsonProperty("itags")
        public List<String> tags;
    }
}
