package ru.yandex.solomon.experiments.gordiychuk.grid;

import java.util.LongSummaryStatistics;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import it.unimi.dsi.fastutil.longs.Long2IntOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import org.apache.commons.math3.stat.descriptive.moment.Mean;
import org.apache.commons.math3.stat.descriptive.moment.StandardDeviation;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeDouble;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.archive.MetricArchiveGeneric;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumn;
import ru.yandex.solomon.model.timeseries.FilteringBeforeAggrGraphDataIterator;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.memState.MetricIdAndData;
import ru.yandex.stockpile.server.shard.iter.SnapshotIterator;

/**
 * @author Vladimir Gordiychuk
 */
public class EstimationActor {
    private static final int MAX_ESTIMATION_INFLIGHT = 500000;
    private static final long TS_BEGIN = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(5);

    private final int shardId;
    private final SnapshotIterator it;
    private final Executor executor;
    private final Consumer<Estimation> consumer;
    private final CompletableFuture<Void> doneFuture;
    private final Metrics readMetrics = new Metrics("stockpile.read");
    private long readMetricsCount = 0;
    private final GaugeDouble progress;

    private final AtomicInteger estimationInFlight = new AtomicInteger();
    private final ActorWithFutureRunner actor;

    public EstimationActor(int shardId, SnapshotIterator it, Consumer<Estimation> consumer, Executor executor) {
        this.shardId = shardId;
        this.it = it;
        this.executor = executor;
        this.consumer = consumer;
        this.progress = MetricRegistry.root().gaugeDouble("estimation.progress", Labels.of("shardId", StockpileShardId.toString(shardId)));
        this.doneFuture = new CompletableFuture<>();
        this.actor = new ActorWithFutureRunner(this::act, executor);
    }

    public CompletableFuture<Void> run() {
        actor.schedule();
        return doneFuture;
    }

    private CompletableFuture<Void> act() {
        if (estimationInFlight.get() > MAX_ESTIMATION_INFLIGHT) {
            return CompletableFuture.completedFuture(null);
        }

        if (doneFuture.isDone()) {
            return CompletableFuture.completedFuture(null);
        }

        return it.next().thenAccept(this::asyncEstimateGrid);
    }

    private boolean isIgnore(MetricIdAndData metric) {
        if (metric.archive().getOwnerProjectIdOrUnknown() != EProjectId.SOLOMON) {
            return true;
        }

        if (metric.archive().getRecordCount() <= 5) {
            return true;
        }

        return metric.archive().hasColumn(StockpileColumn.MERGE);
    }

    private void asyncEstimateGrid(@Nullable MetricIdAndData metric) {
        if (metric == null) {
            if (estimationInFlight.get() == 0) {
                doneFuture.complete(null);
                progress.set(100);
            }
            return;
        }

        this.readMetrics.add(metric.archive());
        this.progress.set(readMetricsCount++ * 100. / it.size());
        if ((readMetricsCount % 10_000) == 0) {
            System.out.println("Estimation shardId " + shardId + " progress: " + String.format("%.2f%%", progress.get()));
        }

        if (isIgnore(metric)) {
            actor.schedule();
            return;
        }

        estimationInFlight.incrementAndGet();
        CompletableFuture.runAsync(() -> {
            try {
                var estimation = estimateGrid(metric);
                if (estimation != null) {
                    consumer.accept(estimation);
                }
            } catch (Throwable e) {
                doneFuture.completeExceptionally(e);
            } finally {
                estimationInFlight.decrementAndGet();
                actor.schedule();
            }
        }, executor);
        actor.schedule();
    }

    @Nullable
    private Estimation estimateGrid(MetricIdAndData metric) {
        LongArrayList tsDeltas = deltaTime(metric.archive());
        if (tsDeltas.size() == 0) {
            return null;
        }

        Long2IntOpenHashMap distribution = new Long2IntOpenHashMap();
        StandardDeviation std = new StandardDeviation();
        Mean mean = new Mean();
        LongSummaryStatistics stats = new LongSummaryStatistics();

        {
            var it = tsDeltas.iterator();
            while (it.hasNext()) {
                long value = it.nextLong();
                distribution.addTo(value, 1);
                std.increment(value);
                mean.increment(value);
                stats.accept(value);
            }
        }

        long dominant = 0;
        {
            int dominantCount = 0;
            var it = distribution.long2IntEntrySet().fastIterator();
            while (it.hasNext()) {
                var entry = it.next();
                if (entry.getIntValue() > dominantCount) {
                    dominantCount = entry.getIntValue();
                    dominant = entry.getLongKey();
                }
            }
        }
        var estimation = new Estimation();
        estimation.numId = metric.archive().getOwnerShardId();
        estimation.shardId = shardId;
        estimation.localId = metric.localId();
        estimation.min = stats.getMin();
        estimation.max = stats.getMax();
        estimation.mean = mean.getResult();
        estimation.std = std.getResult();
        estimation.dominant = dominant;
        return estimation;
    }

    private LongArrayList deltaTime(MetricArchiveGeneric archive) {
        LongArrayList result = new LongArrayList(archive.getRecordCount());
        var it = FilteringBeforeAggrGraphDataIterator.of(TS_BEGIN, archive.iterator());
        var point = RecyclableAggrPoint.newInstance();
        try {
            if (!it.next(point)) {
                return result;
            }

            long prevTsMillis = point.tsMillis;
            while (it.next(point)) {
                long delta = point.tsMillis - prevTsMillis;
                result.add(delta);
                prevTsMillis = point.tsMillis;
            }
        } finally {
            point.recycle();
        }
        return result;
    }

    private static class Metrics {
        final Rate bytes;
        final Rate metrics;
        final Rate records;

        Metrics(String prefix) {
            var registry = MetricRegistry.root();
            bytes = registry.rate(prefix + ".bytes");
            metrics = registry.rate(prefix + ".metrics");
            records = registry.rate(prefix + ".records");
        }

        void add(MetricArchiveImmutable archive) {
            bytes.add(archive.bytesCount());
            records.add(archive.getRecordCount());
            metrics.add(1);
        }
    }
}
