package ru.yandex.solomon.experiments.gordiychuk.grid;

import java.nio.file.Path;
import java.time.Duration;
import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.LongStream;

import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import org.apache.logging.log4j.Level;

import ru.yandex.kikimr.client.KikimrGrpcTransport;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.client.kv.KikimrKvClientImpl;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.JvmThreads;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.experiments.gordiychuk.recovery.MetricsPushScheduler;
import ru.yandex.solomon.experiments.gordiychuk.recovery.stockpile.StockpileShardReader;
import ru.yandex.solomon.main.logger.LoggerConfigurationUtils;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.NettyUtils;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.kikimrKv.ShardIdMapToLong;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.experiments.gordiychuk.grid.IoUtils.mkdir;

/**
 * @author Vladimir Gordiychuk
 */
public class EstimationCli implements AutoCloseable {
    private final static int MAX_SHARD_INFLIGHT = 100;
    private final ShardIdMapToLong mapping;
    private final ExecutorService executor;
    private final KikimrKvClient kvClient;
    private final StockpileShardReader reader;
    private final Estimator estimator;

    public EstimationCli(Path path, SolomonCluster cluster) {
        this.executor = new ForkJoinPool(Runtime.getRuntime().availableProcessors());
        this.kvClient = new KikimrKvClientImpl(new KikimrGrpcTransport(
                cluster.addressesKikimrGrpc(),
                26 << 20,
                Duration.ofSeconds(5),
                Duration.ofSeconds(60),
                NettyUtils.createEventLoopGroup("io", Runtime.getRuntime().availableProcessors()),
                executor));
        this.reader = new StockpileShardReader(kvClient);
        JvmThreads.addExecutorMetrics("CpuLowPriority", executor, MetricRegistry.root());
        this.mapping = new ShardIdMapToLong(kvClient.resolveKvTablets(cluster.getSolomonVolumePath()).join());
        this.estimator = new Estimator(executor, reader, mapping, path);
    }

    public static void main(String[] args) {
        LoggerConfigurationUtils.simpleLogger(Level.INFO);
        MetricsPushScheduler.schedulePush();
        var path = Path.of(args[0]).resolve("shards");
        mkdir(path);
        try (var cli = new EstimationCli(path, SolomonCluster.PROD_STORAGE_SAS)) {
            cli.process(args[1]).join();
            System.out.println("Success: " + Arrays.toString(args));
            System.exit(0);
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    public CompletableFuture<Void> process(String target) {
        if ("local".equals(target)) {
            return processLocalShards();
        } else if ("all".equals(target)) {
            return processShards(mapping.shardIdStream().toArray());
        } else if ("clean".equals(target)) {
            return LongStream.of(mapping.getValues())
                    .mapToObj(reader::dropBackup)
                    .collect(collectingAndThen(toList(), CompletableFutures::allOfVoid));
        } else {
            int shardId = StockpileShardId.parse(target);
            return processShard(shardId);
        }
    }

    public CompletableFuture<Void> processLocalShards() {
        int[] localShardIds;
        try (var kvClient = KikimrHelper.createKvClient("localhost")) {
            LongOpenHashSet set = new LongOpenHashSet(mapping.getValues());
            var localTabletIds = new LongOpenHashSet(LongStream.of(kvClient.findTabletsOnLocalhost().join())
                    .filter(set::contains)
                    .toArray());
            localShardIds = mapping.shardIdStream()
                    .filter(shardId -> localTabletIds.contains(mapping.get(shardId)))
                    .toArray();
        }

        return processShards(localShardIds);
    }

    public CompletableFuture<Void> processShards(int[] shardsIds) {
        var progressMetric = MetricRegistry.root().gaugeDouble("progress");
        AtomicInteger cursor = new AtomicInteger();
        AtomicInteger completed = new AtomicInteger();
        AsyncActorBody body = () -> {
            int index = cursor.getAndIncrement();
            if (index >= shardsIds.length) {
                progressMetric.set(100);
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            return processShard(shardsIds[index]).thenRun(() -> {
                progressMetric.set(completed.incrementAndGet() * 100. / shardsIds.length);
                String progress = String.format("%.2f%%", progressMetric.get());
                System.out.println("Processing at " + HostUtils.getShortName() + " progress: " + progress);
            });
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, executor, MAX_SHARD_INFLIGHT);
        return runner.start()
                .whenComplete((ignore, e) -> {
                    System.out.println("Complete shards processing");
                });
    }

    public CompletableFuture<Void> processShard(int shardId) {
        System.out.println("Start processing shardId " + shardId + " tabletId " + mapping.get(shardId));
        return estimator.run(shardId)
                .whenComplete((ignore, e) -> {
                    MetricRegistry.root().counter("stockpile.estimation.shards.count").inc();
                    System.out.println("Complete estimation shardId " + shardId);
                });
    }

    @Override
    public void close() {
        reader.close();
        kvClient.close();
        executor.shutdownNow();
    }
}
