package ru.yandex.solomon.experiments.gordiychuk.grid;

import java.io.BufferedWriter;
import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

import static ru.yandex.solomon.experiments.gordiychuk.grid.IoUtils.gzWriter;

/**
 * @author Vladimir Gordiychuk
 */
public class EstimationWriter implements AutoCloseable {
    private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();
    private final BufferedWriter writer;
    private final ActorRunner actor;
    private final GaugeInt64 queueSize;
    private final ConcurrentLinkedQueue<String> queue = new ConcurrentLinkedQueue<>();
    private volatile boolean done;

    public EstimationWriter(Path path, Executor executor) {
        this.writer = gzWriter(path);
        this.queueSize = MetricRegistry.root().gaugeInt64("writer.queue.size");
        this.actor = new ActorRunner(this::act, executor);
    }

    public CompletableFuture<Void> done() {
        this.done = true;
        actor.schedule();
        return doneFuture;
    }

    public CompletableFuture<Void> start() {
        return doneFuture;
    }

    public void enqueue(Estimation estimation) {
        if (done) {
            throw new IllegalStateException("Already done");
        }

        queueSize.add(1);
        queue.add(prepareLine(estimation));
        actor.schedule();
    }

    private String prepareLine(Estimation estimation) {
        return Integer.toUnsignedString(estimation.numId) +
                "|" +
                StockpileLocalId.toString(estimation.localId) +
                "|" +
                estimation.min +
                "|" +
                estimation.max +
                "|" +
                estimation.mean +
                "|" +
                estimation.std +
                "|" +
                estimation.dominant +
                "\n";
    }

    public void act() {
        try {
            if (queue.isEmpty() && done) {
                writer.flush();
                doneFuture.complete(null);
                return;
            }

            String estimation;
            while ((estimation = queue.poll()) != null) {
                queueSize.add(-1);
                writer.write(estimation);
            }
        } catch (Throwable e) {
            doneFuture.completeExceptionally(e);
        }
    }

    @Override
    public void close() {
        try {
            writer.close();
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }
}
