package ru.yandex.solomon.experiments.gordiychuk.json;

import java.io.IOException;
import java.net.URL;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.io.Resources;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.Threads;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.string.StringLabelAllocator;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.experiments.gordiychuk.AsyncProfiler;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.metrics.parser.MetricConsumer;
import ru.yandex.solomon.metrics.parser.TreeParser;
import ru.yandex.solomon.metrics.parser.TreeParser.ErrorListenerIgnore;
import ru.yandex.solomon.metrics.parser.TreeParser.FormatListenerIgnore;
import ru.yandex.solomon.metrics.parser.json.TreeParserJson;

/*

Before changes, no label cache
Benchmark                               Mode  Cnt    Score   Error  Units
TreeParserJsonBenchmark.parseIntern    thrpt   10  196.734 ± 6.251  ops/s
TreeParserJsonBenchmark.parseNoIntern  thrpt   10  254.278 ± 2.353  ops/s

After changes, with label cache
Benchmark                                      Mode  Cnt    Score   Error  Units
TreeParserJsonBenchmark.parseIntern           thrpt   10  310.636 ± 3.926  ops/s
TreeParserJsonBenchmark.parseNoIntern         thrpt   10  332.855 ± 3.586  ops/s

 */

/**
 * @author Vladimir Gordiychuk
 */
@Fork(value = 1)
@Measurement(iterations = 10, time = 2, timeUnit = TimeUnit.SECONDS)
@Warmup(iterations = 10, time = 2, timeUnit = TimeUnit.SECONDS)
@State(Scope.Thread)
@Threads(1) //current test not support concurrent execution
@BenchmarkMode(Mode.Throughput)
@OutputTimeUnit(TimeUnit.SECONDS)
public class TreeParserJsonBenchmark {

    private ByteBuf buffer;
    private LabelAllocator allocator;
    private TreeParser parserIntern;
    private TreeParser parserNoIntern;
    private MetricCounter consumer;

    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
                .include(TreeParserJsonBenchmark.class.getName())
                .detectJvmArgs()
                .addProfiler(AsyncProfiler.class)
                .build();

        new Runner(opt).run();
    }

    private static ByteBuf loadBytes1() {
        try {
            URL resource = TreeParserJsonBenchmark.class.getResource("k8s_sample.json");
            return Unpooled.wrappedBuffer(Resources.toByteArray(resource));
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }

    @Setup
    public void setUp() {
        buffer = loadBytes1();
        allocator = new InterningLabelAllocator();
        parserIntern = new TreeParserJson(allocator, "");
        parserNoIntern = new TreeParserJson(StringLabelAllocator.SELF, "");
        consumer = new MetricCounter();
    }

    @Benchmark
    public long parseIntern() {
        parserIntern.parseAndProcess(Labels.of(), buffer, consumer, ErrorListenerIgnore.I, FormatListenerIgnore.I, false);
        return consumer.count;
    }

    @Benchmark
    public long parseNoIntern() {
        parserNoIntern.parseAndProcess(Labels.of(), buffer, consumer, ErrorListenerIgnore.I, FormatListenerIgnore.I, false);
        return consumer.count;
    }

    @ParametersAreNonnullByDefault
    private static final class MetricCounter implements MetricConsumer {
        long count = 0;

        @Override
        public void onMetricBegin(MetricType type, Labels labels, boolean memOnly) {
            count++;
        }

        @Override
        public void onPoint(long tsMillis, double value) {
        }

        @Override
        public void onPoint(long tsMillis, long value) {
        }

        @Override
        public void onTimeSeries(TimeSeries timeSeries) {
        }
    }
}
