package ru.yandex.solomon.experiments.gordiychuk.processing;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;

import javax.annotation.Nullable;

import com.google.protobuf.ByteString;
import io.netty.buffer.ByteBufAllocator;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.slog.LogsIndex;
import ru.yandex.solomon.slog.LogsIndexSerializer;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.solomon.util.protobuf.ByteStrings;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class LogWriter implements AutoCloseable {
    private final long tableId;
    private final KikimrKvClient kvClient;
    private ArrayListLockQueue<Request> queue;
    private ActorWithFutureRunner actor;
    private volatile boolean closed;

    public LogWriter(long tableId, KikimrKvClient kvClient) {
        this.tableId = tableId;
        this.kvClient = kvClient;
        this.queue = new ArrayListLockQueue<>();
        this.actor = new ActorWithFutureRunner(this::act, ForkJoinPool.commonPool());
    }

    public CompletableFuture<Void> enqueue(Log entry) {
        var req = new Request(entry);
        queue.enqueue(req);
        actor.schedule();
        return req.future;
    }

    public CompletableFuture<Void> act() {
        var requests = queue.dequeueAll();

        if (requests.isEmpty()) {
            return completedFuture(null);
        }

        if (closed) {
            Throwable e = new RuntimeException("Already closed");
            for (var req : requests) {
                req.complete(e);
            }
        }

        ByteString meta = ByteString.EMPTY;
        ByteString data = ByteString.EMPTY;
        var logsIndex = new LogsIndex(requests.size());

        for (var req : requests) {
            var entry = req.entry;
            meta = meta.concat(ByteStrings.fromByteBuf(entry.meta));
            data = data.concat(ByteStrings.fromByteBuf(entry.data));
            logsIndex.add(entry.numId, entry.meta.readableBytes(), entry.data.readableBytes());
        }
        var serializedLogsIndex = LogsIndexSerializer.serialize(ByteBufAllocator.DEFAULT, logsIndex);

        String suffix = System.currentTimeMillis() + "." + HostUtils.getFqdn();
        return CompletableFuture.allOf(
            write("tmp.index." + suffix, ByteStrings.fromByteBuf(serializedLogsIndex)),
            write("tmp.meta." + suffix, meta),
            write("tmp.data." + suffix, data))
            .thenCompose(ignore -> rename(List.of(
                new KikimrKvClient.Rename("tmp.index." + suffix, "index." + suffix),
                new KikimrKvClient.Rename("tmp.meta." + suffix, "meta." + suffix),
                new KikimrKvClient.Rename("tmp.data." + suffix, "data." + suffix)
            )))
            .whenCompleteAsync((ignore, e) -> {
                serializedLogsIndex.release();
                for (var req : requests) {
                    req.complete(e);
                }
            });
    }

    private CompletableFuture<Void> rename(List<KikimrKvClient.Rename> renames) {
        return kvClient.writeAndRename(tableId, 0, List.of(), renames, 0);
    }

    private CompletableFuture<Void> write(String name, ByteString bytes) {
        return kvClient.writeMulti(tableId, 0, List.of(prepareWrite(name, bytes)), 0);
    }

    private KikimrKvClient.Write prepareWrite(String name, ByteString bytes) {
        return new KikimrKvClient.Write(
            name,
            bytes,
            MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
            MsgbusKv.TKeyValueRequest.EPriority.REALTIME
        );
    }

    @Override
    public void close() {
        closed = true;
        actor.schedule();
    }

    private static class Request {
        private final Log entry;
        private final CompletableFuture<Void> future;

        public Request(Log entry) {
            this.entry = entry;
            this.future = new CompletableFuture<>();
        }

        public void complete(@Nullable Throwable e) {
            entry.close();
            if (e != null) {
                future.completeExceptionally(e);
            } else {
                future.complete(null);
            }
        }
    }
}
