package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import io.netty.util.Recycler;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class AsyncMappingWriter {
    private final Path file;
    private final ConcurrentLinkedQueue<Entry> queue = new ConcurrentLinkedQueue<>();
    private final MappingWriter writer;
    private volatile boolean complete;
    private CompletableFuture<Void> doneFuture = new CompletableFuture<>();
    private ActorRunner actor;
    private Counter writeRecords = MetricRegistry.root().counter("merge.metabase.write.records");
    private GaugeInt64 queueSize = MetricRegistry.root().gaugeInt64("merge.write.queue.size");
    private AtomicReference<CompletableFuture<Void>> notFull = new AtomicReference<>(new CompletableFuture<>());
    private AtomicInteger limit = new AtomicInteger(1000);
    private AtomicInteger size = new AtomicInteger();

    public AsyncMappingWriter(Path file) {
        try {
            Files.createDirectories(file.getParent());
            this.file = file;
            this.writer = new MappingWriter(file);
            this.actor = new ActorRunner(this::act, ForkJoinPool.commonPool());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public CompletableFuture<Void> write(Record local, Record remote) {
        try {
            if (doneFuture.isCompletedExceptionally()) {
                return doneFuture;
            }

            queueSize.add(1);
            queue.add(Entry.of(local, remote));
            if (queueSize.get() >= limit.get()) {
                CompletableFuture<Void> prevSync, next;
                do {
                    prevSync = notFull.get();
                    if (!prevSync.isDone()) {
                        next = prevSync;
                        break;
                    }
                    next = new CompletableFuture<>();
                } while (!notFull.compareAndSet(prevSync, next));

                actor.schedule();
                return next;
            } else {
                actor.schedule();
                return CompletableFuture.completedFuture(null);
            }
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    public void complete() {
        complete = true;
        actor.schedule();
    }

    public CompletableFuture<Void> doneFuture() {
        return doneFuture;
    }

    private void act() {
        if (doneFuture.isDone()) {
            notFullNow();
            return;
        }

        try {
            Entry entry;
            while ((entry = queue.poll()) != null) {
                writer.write(entry);
                writeRecords.inc();
                queueSize.add(-1);
                size.decrementAndGet();
                entry.recycle();
                notFullNow();
            }

            if (!complete) {
                notFullNow();
                limit.incrementAndGet();
                return;
            }

            if (queue.isEmpty()) {
                System.out.println("Complete write to " + file);
                writer.close();
                doneFuture.complete(null);
            }
        } catch (Throwable e) {
            doneFuture.completeExceptionally(e);
        }
    }

    private void notFullNow() {
        var sync = notFull.get();
        if (!sync.isDone()) {
            sync.completeAsync(() -> null);
        }
    }

    private static class Entry extends MappingRecord {
        private static final Recycler<Entry> RECYCLER = new Recycler<>() {
            protected Entry newObject(Handle<Entry> handle) {
                return new Entry(handle);
            }
        };

        private final Recycler.Handle<Entry> handle;

        private Entry(Recycler.Handle<Entry> handle) {
            this.handle = handle;
        }

        public static Entry of(Record local, Record remote) {
            var entry = RECYCLER.get();
            entry.localLocalId = local.localId;
            entry.remoteShardId = remote.shardId;
            entry.remoteLocalId = remote.localId;
            return entry;
        }

        public void recycle() {
            this.handle.recycle(this);
        }
    }
}
