package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

/**
 * @author Vladimir Gordiychuk
 */
public class AsyncMappingWriters implements AutoCloseable {
    private final Path root;
    private final ReentrantReadWriteLock rwLock = new ReentrantReadWriteLock();
    private final Int2ObjectMap<AsyncMappingWriter> mapping;

    public AsyncMappingWriters(Path root) {
        this.root = root;
        this.mapping = new Int2ObjectOpenHashMap<>();
    }

    public CompletableFuture<Void> map(Record local, Record remote) {
        var shard = writerOrNull(local.shardId);
        if (shard == null) {
            shard = initWriter(local.shardId);
        }
        return shard.write(local, remote);
    }

    private AsyncMappingWriter writerOrNull(int shardId) {
        var lock = rwLock.readLock();
        lock.lock();
        try {
            return mapping.get(shardId);
        } finally {
            lock.unlock();
        }
    }

    private AsyncMappingWriter initWriter(int shardId) {
        var lock = rwLock.writeLock();
        lock.lock();
        try {
            AsyncMappingWriter writer = mapping.get(shardId);
            if (writer == null) {
                writer = new AsyncMappingWriter(root.resolve("mapping").resolve("" + shardId));
                mapping.put(shardId, writer);
            }
            return writer;
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void close() {
        var lock = rwLock.writeLock();
        lock.lock();
        try {
            mapping.values().forEach(AsyncMappingWriter::complete);
            mapping.values().forEach(writer -> writer.doneFuture().join());
        } finally {
            lock.unlock();
        }
    }
}
