package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.util.ExceptionUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class AsyncRecordReader {
    private static final long LIMIT = 10_000;
    private final RecordIterator iterator;
    private ConcurrentLinkedQueue<Record> queue = new ConcurrentLinkedQueue<>();
    private volatile boolean end;
    private ActorRunner actor;
    private Counter readRecords = MetricRegistry.root().counter("merge.metabase.read.records");
    private GaugeInt64 globalQueueSize = MetricRegistry.root().gaugeInt64("merge.read.queue.size");
    private AtomicInteger queueSize = new AtomicInteger();
    private volatile CompletableFuture<Void> notEmpty = new CompletableFuture<>();

    public AsyncRecordReader(Path file) {
        this.iterator = new RecordIterator(file);
        this.actor = new ActorRunner(this::act, ForkJoinPool.commonPool());
    }

    public CompletableFuture<Record> next() {
        try {
            if (end) {
                var record = queue.poll();
                if (record != null) {
                    globalQueueSize.add(-1);
                    queueSize.decrementAndGet();
                }
                return CompletableFuture.completedFuture(record);
            }

            actor.schedule();
            var record = queue.poll();
            if (record != null) {
                globalQueueSize.add(-1);
                queueSize.decrementAndGet();
                return CompletableFuture.completedFuture(record);
            } else {
                var sync = new CompletableFuture<Void>();
                notEmpty = sync;
                actor.schedule();
                return sync.thenComposeAsync(ignore -> next());
            }
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private void act() {
        try {
            if (end) {
                notEmpty.complete(null);
                return;
            }

            while (true) {
                if (queueSize.get() >= LIMIT) {
                    return;
                }

                var record = iterator.next();
                if (record == null) {
                    System.out.println("Read file " + iterator.file + " done");
                    iterator.close();
                    end = true;
                    notEmpty.complete(null);
                    return;
                }

                readRecords.inc();
                globalQueueSize.add(1);
                queueSize.incrementAndGet();
                queue.add(record);
                notEmpty.complete(null);
            }
        } catch (Throwable e) {
            ExceptionUtils.uncaughtException(e);
        }
    }
}
