package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.nio.file.Path;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class AsyncRecordWriter {
    private final RecordWriter writer;
    private final AtomicInteger write = new AtomicInteger();
    private final ConcurrentLinkedQueue<Record> queue = new ConcurrentLinkedQueue<>();
    private final ActorRunner actor;
    private final CompletableFuture<Void> doneFuture;
    private volatile boolean complete;
    private final GaugeInt64 queueSize = MetricRegistry.root().gaugeInt64("write.queue.size");

    public AsyncRecordWriter(Path file, Executor executor) {
        this.writer = new RecordWriter(file);
        this.doneFuture = new CompletableFuture<>();
        this.actor = new ActorRunner(this::act, executor);
    }

    public void complete() {
        complete = true;
        actor.schedule();
    }

    public void add(Record record) {
        queueSize.add(1);
        queue.add(record);
        actor.schedule();
    }

    public CompletableFuture<Void> doneFuture() {
        return doneFuture;
    }

    private void act() {
        if (doneFuture.isDone()) {
            return;
        }

        try {
            Record record;
            while ((record = queue.poll()) != null) {
                try {
                    queueSize.add(-1);
                    writer.write(record);
                    write.incrementAndGet();
                } catch (Throwable e) {
                    doneFuture.completeExceptionally(e);
                    return;
                }
            }

            if (!complete) {
                return;
            }

            if (queue.isEmpty()) {
                writer.close();
                doneFuture.complete(null);
            }
        } catch (Throwable e) {
            writer.close();
            doneFuture.completeExceptionally(e);
        }
    }
}
