package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.concurrent.ForkJoinPool;

import ru.yandex.solomon.experiments.gordiychuk.recovery.graphite.DumpMetricsFromGraphiteTask;
import ru.yandex.solomon.experiments.gordiychuk.recovery.metabase.DumpMetricsFromMetabaseTask;
import ru.yandex.solomon.experiments.gordiychuk.recovery.metabase.MetabaseFileMetricsMerger;
import ru.yandex.solomon.experiments.gordiychuk.recovery.metabase.RestoreNonExistsMetrics;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Vladimir Gordiychuk
 */
public class MetabaseRecovery {
    public static void main(String[] args) {
        try {
            MetricsPushScheduler.schedulePush();
            switch (args[0]) {
                // dump metabase PROD_STORAGE_SAS PROD_STORAGE_VLA ./dump
                // dump graphite PROD_STORAGE_SAS PROD_STORAGE_VLA ./dump
                case "dump":
                    dump(Arrays.copyOfRange(args, 1, args.length));
                    break;
                // merge metabase PROD_STORAGE_SAS PROD_STORAGE_VLA ./dump
                case "merge":
                    merge(Arrays.copyOfRange(args, 1, args.length));
                    break;
                // upload PROD_STORAGE_SAS ./dump
                case "upload":
                    upload(Arrays.copyOfRange(args, 1, args.length));
                    break;
                // recovery metabase PROD_STORAGE_VLA ./dump
                case "recovery":
                    recovery(Arrays.copyOfRange(args, 1, args.length));
                    break;
                default:
                    System.err.println("Unknown command: "+ Arrays.toString(args));
                    System.exit(1);
            }

            System.out.println("Success: " + Arrays.toString(args));
            System.exit(0);
        } catch (Throwable e) {
            e.printStackTrace();
            System.exit(1);
        }
    }

    private static void dump(String[] args) throws IOException {
        Path root = Path.of(args[3]);
        Files.createDirectories(root);

        SolomonCluster one = SolomonCluster.valueOf(args[1]);
        SolomonCluster two = SolomonCluster.valueOf(args[2]);
        switch (args[0]) {
            case "metabase": {
                var task = new DumpMetricsFromMetabaseTask(ForkJoinPool.commonPool(), root);
                task.run(one);
                task.run(two);
                break;
            }
            case "graphite": {
                try (var task = new DumpMetricsFromGraphiteTask(root, one, two)) {
                    task.run();
                }
                break;
            }
            default:
                System.err.println("Unknown command: "+ Arrays.toString(args));
                System.exit(1);
        }
    }

    public static void merge(String[] args) throws IOException {
        Path root = Path.of(args[3]);
        Files.createDirectories(root);

        SolomonCluster one = SolomonCluster.valueOf(args[1]);
        SolomonCluster two = SolomonCluster.valueOf(args[2]);
        switch (args[0]) {
            case "metabase":
                MetabaseFileMetricsMerger.merge(
                    root.resolve(one.name().toLowerCase()),
                    root.resolve(two.name().toLowerCase()));
                break;
            default:
                System.err.println("Unknown command: "+ Arrays.toString(args));
                System.exit(1);
        }
    }

    public static void upload(String[] args) throws IOException {
        Path root = Path.of(args[1]);
        Files.createDirectories(root);

        SolomonCluster cluster = SolomonCluster.valueOf(args[0]);
        var path = root.resolve(cluster.name().toLowerCase());
        UploadMappingToShards.upload(path, cluster);
    }

    public static void recovery(String[] args) throws IOException {
        Path root = Path.of(args[2]);
        Files.createDirectories(root);

        SolomonCluster cluster = SolomonCluster.valueOf(args[1]);
        switch (args[0]) {
            case "metabase":
                RestoreNonExistsMetrics.recovery(cluster, root);
                break;
            default:
                System.err.println("Unknown command: "+ Arrays.toString(args));
                System.exit(1);
        }
    }
}
