package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.zip.GZIPOutputStream;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;

/**
 * @author Vladimir Gordiychuk
 */
public class RecordWriter implements AutoCloseable {
    private final Path path;
    private BufferedWriter out;
    private long writeLines;

    public RecordWriter(Path path) {
        try {
            this.path = path;
            Files.createDirectories(path.getParent());
            var fout = Files.newOutputStream(path, StandardOpenOption.CREATE, StandardOpenOption.APPEND);
            var gzip = new GZIPOutputStream(fout);
            var writer = new OutputStreamWriter(gzip, StandardCharsets.UTF_8);
            this.out = new BufferedWriter(writer);
        } catch (IOException e) {
            throw new RuntimeException(path.toString(), e);
        }
    }

    public void write(Record record) {
        try {
            for (int index = 0; index < record.labels.length(); index++) {
                char c = record.labels.charAt(index);
                if (Character.isISOControl(c) || c == '|' || c == '\n' || c == '\t') {
                    return;
                }
            }

            out.write(record.labels);
            out.write('|');
            out.write(Integer.toString(record.flags));
            out.write('|');
            out.write(StockpileShardId.toString(record.shardId));
            out.write('|');
            out.write(StockpileLocalId.toString(record.localId));
            out.newLine();

            if ((++writeLines % 1_000_000) == 0) {
                System.out.println("Write file " + path + " " + DataSize.shortString(writeLines));
            }

        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void close() {
        try {
            out.flush();
            out.close();
            System.out.println("Write " + path + " done");
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
