package ru.yandex.solomon.experiments.gordiychuk.recovery;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.base.Throwables;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.proto.MsgbusKv;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.shard.StockpileShardId;
import ru.yandex.stockpile.kikimrKv.ShardIdMapToLong;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientCounting;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientMetrics;
import ru.yandex.stockpile.kikimrKv.counting.WriteClass;
import ru.yandex.stockpile.server.data.dao.LargeFileWriter;

/**
 * @author Vladimir Gordiychuk
 */
public class UploadMappingToShards {
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
        .withNumRetries(Integer.MAX_VALUE)
        .withDelay(300)
        .withStats((timeSpentMillis, cause) -> {
            System.out.println("Failed on upload, retrying...\n" + Throwables.getStackTraceAsString(cause));
        });

    private static final int MAX_IN_FLIGHT = Runtime.getRuntime().availableProcessors();

    private Path root;
    private String volumePath;
    private KikimrKvClient kvClient;
    private KikimrKvClientCounting kvClientCounting;

    public static void upload(Path root, SolomonCluster cluster) {
        try (var kvClient = KikimrHelper.createKvClient(cluster)) {
            new UploadMappingToShards(root, cluster.getSolomonVolumePath(), kvClient).run();
        }
    }

    public UploadMappingToShards(Path root, String volumePath, KikimrKvClient client) {
        this.root = root;
        this.volumePath = volumePath;
        this.kvClient = client;
        this.kvClientCounting = new KikimrKvClientCounting(kvClient, new KikimrKvClientMetrics(MetricRegistry.root()));
    }

    public void run() {
        var mapping = kvClient.resolveKvTablets(volumePath)
            .thenApply(ShardIdMapToLong::new)
            .join();

        var files = files();
        AtomicInteger index = new AtomicInteger();
        AsyncActorBody body = () -> {
            int i = index.getAndIncrement();
            if (i >= files.size()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var file = files.get(i);
            int shardId = StockpileShardId.parse(file.getFileName().toString());
            var tabletId = mapping.get(shardId);
            return CompletableFuture.supplyAsync(() -> readBytes(file))
                .thenCompose(bytes -> {
                    var writer = new LargeFileWriter(
                        kvClientCounting,
                        tabletId,
                        0,
                        WriteClass.OTHER,
                        MsgbusKv.TKeyValueRequest.EStorageChannel.MAIN,
                        MsgbusKv.TKeyValueRequest.EPriority.BACKGROUND);
                    return retry(() -> writer.write("mapping", bytes));
                })
                .whenComplete((ignore, e) -> {
                    System.out.println("Complete load mapping to tabletId " + tabletId);
                    double progress = i * 100. / files.size();
                    System.out.println("Upload progress: " + String.format("%.2f%%", progress));
                    MetricRegistry.root().gaugeDouble("upload.progress").set(progress);
                });
        };

        new AsyncActorRunner(body, ForkJoinPool.commonPool(), MAX_IN_FLIGHT).start().join();
    }

    private List<Path> files() {
        try {
            return Files.list(root.resolve("mapping"))
                .collect(Collectors.toList());
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    private byte[] readBytes(Path file) {
        try {
            return Files.readAllBytes(file);
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    private static <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return RetryCompletableFuture.runWithRetries(supplier, RETRY_CONFIG);
    }
}
