package ru.yandex.solomon.experiments.gordiychuk.recovery.graphite;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.values.TupleValue;

import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;

/**
 * @author Vladimir Gordiychuk
 */
public class GraphiteMetricsReader {
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT.withMaxDelay(1000);

    private final YdbClient client;

    public GraphiteMetricsReader(YdbClient client) {
        this.client = client;
    }

    private CompletableFuture<Long> getMetricsCount(String path) {
        String query = String.format("""
                --!syntax_v1
                select cast(count(*) as Uint64) from `%s`;
                """, path);
        return client.fluent().execute(query).thenApply(result -> {
            var resultSet = result.expect("success").getResultSet(0);
            if (!resultSet.next()) {
                return 0L;
            }

            return resultSet.getColumn(0).getUint64();
        });
    }

    public CompletableFuture<Void> read(String path, Consumer consumer) {
        AtomicLong read = new AtomicLong();
        return CompletableFuture.completedFuture(null)
                .thenCompose(ignore -> getMetricsCount(path))
                .thenCompose(metrics -> {
                    MetricsReader reader = new MetricsReader();
                    return RetryCompletableFuture.runWithRetries(() -> {
                        var from = reader.lastKey();
                        System.out.println("Read table " + path + " from " + from + "...");

                        return client.fluent().executeOnSession(session -> {
                            var settings = ReadTableSettings.newBuilder()
                                    .orderedRead(true)
                                    .timeout(30, TimeUnit.SECONDS);

                            if (from != null) {
                                settings.fromKeyExclusive(from);
                            }

                            return session.readTable(path, settings.build(), resultSet -> {
                                reader.read(resultSet, consumer);
                                read.addAndGet(resultSet.getRowCount());

                                double progress = read.get() * 100. / metrics;
                                System.out.println("Read table " + path + " " + String.format("%.2f%%", progress));
                            });
                        }).thenAccept(status -> status.expect("can not read table " + path));
                    }, RETRY_CONFIG);
                })
                .thenAccept(unit -> {
                    System.out.println("Read table " + path + " done");
                });
    }

    public interface Consumer {
        void accept(String name, int shardId, long localId);
    }

    /**
     * METRICS READER
     */
    private static final class MetricsReader {
        private int hashIdx = -1;
        private int nameIdx = -1;
        private int shardIdIdx = -1;
        private int localIdIdx = -1;
        private int lastHash;
        private String lastName;

        void read(ResultSetReader rs, Consumer fn) {
            final int rowsCount = rs.getRowCount();
            if (rowsCount == 0) {
                return;
            }

            if (hashIdx == -1) {
                hashIdx = rs.getColumnIndex("hash");
                nameIdx = rs.getColumnIndex("name");
                shardIdIdx = rs.getColumnIndex("shardId");
                localIdIdx = rs.getColumnIndex("localId");
            }

            while (rs.next()) {
                lastHash = (int) rs.getColumn(hashIdx).getUint32();
                lastName = rs.getColumn(nameIdx).getUtf8();

                int shardId = rs.getColumn(shardIdIdx).getInt32();
                long localId = rs.getColumn(localIdIdx).getInt64();
                fn.accept(lastName, shardId, localId);
            }
        }

        public TupleValue lastKey() {
            if (lastName == null) {
                return null;
            }

            return TupleValue.of(
                    uint32(lastHash).makeOptional(),
                    utf8(lastName).makeOptional());
        }
    }
}
