package ru.yandex.solomon.experiments.gordiychuk.recovery.metabase;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import javax.annotation.WillClose;

import com.yandex.ydb.scheme.SchemeOperationProtos.Entry;
import com.yandex.ydb.scheme.SchemeOperationProtos.Entry.Type;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Vladimir Gordiychuk
 */
public class DumpMetricsFromClusterTask {
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT.withNumRetries(1000);
    private static final int MAX_SHARD_INFLIGHT = Runtime.getRuntime().availableProcessors();

    private final String rootSchema;
    private final Path root;
    private final YdbClient client;
    private final Executor executor;

    public DumpMetricsFromClusterTask(SolomonCluster cluster, Executor executor, Path root) {
        this(cluster.kikimrRootPath(), root.resolve(cluster.name().toLowerCase()), YdbHelper.createYdbClient(cluster), executor);
    }

    public DumpMetricsFromClusterTask(String rootSchema, Path root, @WillClose YdbClient client, Executor executor) {
        this.rootSchema = rootSchema;
        this.root = root.resolve("metrics");
        this.client = client;
        this.executor = executor;
        try {
            Files.createDirectories(this.root);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private CompletableFuture<List<String>> listSolomonTables() {
        return RetryCompletableFuture.runWithRetries(() -> {
            return client.scheme.listDirectory(rootSchema + "/Solomon/metrics")
                    .thenApply(resultSet -> {
                        return resultSet.expect("failed list dirs")
                                .getChildren()
                                .stream()
                                .filter(entry -> entry.getType() == Type.TABLE)
                                .map(Entry::getName)
                                .collect(Collectors.toList());
                    });
        }, RETRY_CONFIG);
    }

    public void run() {
        List<String> tablets = listSolomonTables().join();
        AtomicInteger current = new AtomicInteger();
        AsyncActorBody body = () -> {
            int index = current.getAndIncrement();
            if (index >= tablets.size()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var table = tablets.get(index);
            var file = root.resolve(table);
            var task = new DumpMetricsFromShardTask(rootSchema, client, table, executor, file);

            double progress = index * 100. / tablets.size();
            System.out.println("Dump to folder " + root + " progress: " + String.format("%.2f%%", progress));
            MetricRegistry.root().gaugeDouble("dump.metabase.progress").set(progress);
            return task.run();
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, executor, MAX_SHARD_INFLIGHT);
        runner.start().join();
        client.close();
    }
}
