package ru.yandex.solomon.experiments.gordiychuk.recovery.metabase;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.WillClose;

import com.google.common.collect.Sets;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.solomon.experiments.gordiychuk.recovery.AsyncMappingWriters;
import ru.yandex.solomon.experiments.gordiychuk.recovery.AsyncRecordReader;
import ru.yandex.solomon.experiments.gordiychuk.recovery.AsyncRecordWriter;
import ru.yandex.solomon.experiments.gordiychuk.recovery.Record;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public class MetabaseFileMetricsMerger {
    private static final int MAX_SHARD_INFLIGHT = Runtime.getRuntime().availableProcessors();

    public static void merge(Path left, Path right) {
        System.out.println("Starting merge files: [" + left + ", " + right + "]");

        try (AsyncMappingWriters leftMapping = new AsyncMappingWriters(left);
             AsyncMappingWriters rightMapping = new AsyncMappingWriters(right))
        {
            List<String> shards = List.copyOf(Sets.intersection(shards(left), shards(right)));
            AtomicInteger current = new AtomicInteger();
            AsyncActorBody body = () -> {
                int index = current.getAndIncrement();
                if (index >= shards.size()) {
                    System.out.println("done all shards");
                    return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                }

                var shardId = shards.get(index);
                System.out.println("Starting merge files:" + shardId);
                return merge(new Target(left, shardId, leftMapping), new Target(right, shardId, rightMapping))
                    .whenComplete((ignore, e) -> {
                        if (e != null) {
                            e.printStackTrace();
                        }

                        System.out.println("Complete merge shard " + shardId);
                        String progress = String.format("%.2f%%", index * 100. / shards.size());
                        System.out.println("Merge progress: " + progress);
                });
            };

            AsyncActorRunner runner = new AsyncActorRunner(body, ForkJoinPool.commonPool(), MAX_SHARD_INFLIGHT);
            runner.start().join();
            System.out.println("Merge success");
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static Set<String> shards(Path path) throws IOException {
        return Files.list(path.resolve("metrics"))
            .map(file -> file.getFileName().toString())
            .collect(Collectors.toSet());
    }

    public static CompletableFuture<Void> merge(@WillClose Target left, @WillClose Target right) {
        return new MergeTask(left, right).start();
    }

    private static class Target implements AutoCloseable {
        final Path root;
        final String shardId;
        Record record;
        AsyncRecordReader reader;
        @Nullable
        AsyncRecordWriter notExists;
        AsyncMappingWriters mapping;

        public Target(Path root, String shardId, AsyncMappingWriters mapping) {
            this.root = root;
            this.shardId = shardId;
            this.reader = new AsyncRecordReader(root.resolve("metrics").resolve(shardId));
            this.notExists = new AsyncRecordWriter(root.resolve("absent").resolve(shardId), ForkJoinPool.commonPool());
            this.mapping = mapping;
        }

        public CompletableFuture<Void> fetch() {
            if (record == null) {
                return reader.next()
                    .thenAccept(r -> {
                        record = r;
                    });
            }

            return CompletableFuture.completedFuture(null);
        }

        public void absent(Record record) {
            if (notExists == null) {
                notExists = new AsyncRecordWriter(root.resolve("absent").resolve(shardId), ForkJoinPool.commonPool());
            }
            notExists.add(record);
        }

        public CompletableFuture<Void> map(Record remote) {
            try {
                return mapping.map(this.record, remote);
            } finally {
                record = null;
            }
        }

        public CompletableFuture<Void> map(Record local, Record remote) {
            return mapping.map(local, remote);
        }

        public void clear() {
            record = null;
        }

        @Override
        public void close() {
            if (notExists != null) {
                notExists.complete();
                notExists.doneFuture().join();
            }
        }
    }

    private static class MergeTask {
        @WillClose
        private final Target left;
        @WillClose
        private final Target right;
        private final CompletableFuture<Void> doneFuture;
        private final ActorWithFutureRunner actor;

        public MergeTask(@WillClose Target left, @WillClose Target right) {
            this.left = left;
            this.right = right;
            this.doneFuture = new CompletableFuture<>();
            this.actor = new ActorWithFutureRunner(this::act, ForkJoinPool.commonPool());
        }

        private CompletableFuture<Void> act() {
            if (doneFuture.isDone()) {
                return doneFuture;
            }

            return read()
                .thenCompose(ignore -> merge())
                .thenRun(actor::schedule);
        }

        private CompletableFuture<Void> read() {
            return CompletableFuture.allOf(left.fetch(), right.fetch());
        }

        private CompletableFuture<Void> merge() {
            if (right.record == null && left.record == null) {
                right.close();
                left.close();
                doneFuture.complete(null);
                return doneFuture;
            }

            if (left.record == null) {
                left.absent(right.record);
                return right.map(right.record);
            } else if (right.record == null) {
                right.absent(left.record);
                return left.map(left.record);
            }

            int compare = left.record.compareTo(right.record);
            if (compare < 0) {
                right.absent(left.record);
                return left.map(left.record);
            } else if (compare > 0) {
                left.absent(right.record);
                return right.map(right.record);
            } else if (left.record.flags != right.record.flags) {
                left.clear();
                right.clear();
                return CompletableFuture.completedFuture(null);
            } else {
                var one = left.map(left.record, right.record);
                var two = right.map(right.record, left.record);
                left.clear();
                right.clear();
                return CompletableFuture.allOf(one, two);
            }
        }

        public CompletableFuture<Void> start() {
            actor.schedule();
            return doneFuture;
        }
    }
}
