package ru.yandex.solomon.experiments.gordiychuk.recovery.metabase;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import javax.annotation.WillNotClose;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public class RestoreNonExistsMetrics {
    private static final int MAX_SHARD_INFLIGHT = Runtime.getRuntime().availableProcessors();

    private final String schemaRoot;
    @WillNotClose
    private final YdbClient ydb;
    private final Path folder;
    @WillNotClose
    private final ExecutorService executor;
    private final List<String> shards;

    public RestoreNonExistsMetrics(String schemaRoot, @WillNotClose YdbClient ydb, Path folder, @WillNotClose ExecutorService executor) {
        try {
            this.schemaRoot = schemaRoot;
            this.ydb = ydb;
            this.folder = folder;
            this.executor = executor;
            this.shards = Files.list(folder.resolve("absent"))
                .map(path -> path.getFileName().toString())
                .collect(Collectors.toList());
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void recovery(SolomonCluster cluster, Path path) {
        try (var ydb = YdbHelper.createYdbClient(cluster)) {
            new RestoreNonExistsMetrics(cluster.kikimrRootPath(), ydb, path, ForkJoinPool.commonPool()).run().join();
        }
    }

    public CompletableFuture<Void> run() {
        System.out.println("Start restore metrics from folder " + folder);

        AtomicInteger cursor = new AtomicInteger();
        AsyncActorBody body = () -> {
            int index = cursor.getAndIncrement();
            if (index >= shards.size()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var shardId = shards.get(index);
            double progress = index * 100. / shards.size();
            MetricRegistry.root().gaugeDouble("metabase.recovery.shard.progress").set(progress);
            System.out.println("Recovery metabase progress: " + String.format("%.2f%%", progress));

            String tablePath = schemaRoot + "/Solomon/metrics/" + shardId;
            return new RestoreNonExistsMetricsTask(tablePath, ydb, folder.resolve("absent").resolve(shardId), executor).start();
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, executor, MAX_SHARD_INFLIGHT);
        return runner.start()
            .whenComplete((ignore, e) -> {
                System.out.println("Complete metabase recovery");
            });
    }
}
