package ru.yandex.solomon.experiments.gordiychuk.recovery.metabase;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.WillClose;
import javax.annotation.WillNotClose;

import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructValue;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.experiments.gordiychuk.recovery.Record;
import ru.yandex.solomon.experiments.gordiychuk.recovery.RecordIterator;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class RestoreNonExistsMetricsTask {
    private static final int BATCH_SIZE = 1000;

    private final String tablePath;
    @WillNotClose
    private final YdbClient ydb;
    private final Path file;
    @WillClose
    private final RecordIterator it;
    private final ActorWithFutureRunner actor;
    private final CompletableFuture<Void> doneFuture = new CompletableFuture<>();

    public RestoreNonExistsMetricsTask(SolomonCluster cluster, Path file, ExecutorService executor) {
        this(cluster.kikimrRootPath(), YdbHelper.createYdbClient(cluster), file, executor);
    }

    public RestoreNonExistsMetricsTask(String tablePath, @WillNotClose YdbClient ydb, Path file, @WillNotClose ExecutorService executor) {
        this.tablePath = tablePath;
        this.ydb = ydb;
        this.file = file;
        this.it = new RecordIterator(file);
        this.actor = new ActorWithFutureRunner(this::act, executor);
    }

    public CompletableFuture<Void> start() {
        System.out.println("Start restore metrics from file " + file);
        actor.schedule();
        return doneFuture;
    }

    private CompletableFuture<Void> act() {
        if (doneFuture.isDone()) {
            return completedFuture(null);
        }

        try {
            Record record;
            List<Record> batch = new ArrayList<>(BATCH_SIZE);
            while ((record = it.next()) != null) {
                batch.add(record);
                if (batch.size() >= BATCH_SIZE) {
                    return replace(batch);
                }
            }

            if (!batch.isEmpty()) {
                return replace(batch);
            }

            it.close();
            System.out.println("Complete recovery: " + tablePath);
            MetricRegistry.root().counter("metabase.recovery.shards.count").inc();
            doneFuture.complete(null);
        } catch (Throwable e) {
            doneFuture.completeExceptionally(e);
        }

        return completedFuture(null);
    }

    private CompletableFuture<Void> replace(List<Record> records) {
        StructValue[] structs = records.stream().map(this::toStruct).toArray(StructValue[]::new);
        ListValue listStructs = ListType.of(structs[0].getType()).newValueOwn(structs);
        Params params = Params.of("$input", listStructs);

        String query = String.format("""
                --!syntax_v1
                DECLARE $input AS List<Struct<hash:Uint32,labels:Utf8,shardId:Uint32,localId:Uint64,createdSeconds:Uint64,flags:Uint32>>;
                UPSERT INTO `%s`
                SELECT * FROM AS_TABLE($input);
                """, tablePath);

        int size = records.size();
        return ydb.fluent()
            .retryForever()
            .execute(query, params)
            .thenApply(r -> r.expect("success"))
            .handle((status, e) -> {
                if (ifTableNotExists(e)) {
                    doneFuture.complete(null);
                } else {
                    if (e != null) {
                        doneFuture.completeExceptionally(e);
                    }

                    MetricRegistry.root().counter("metabase.recovery.write.metrics").add(size);
                    actor.schedule();
                }
                return null;
            });
    }

    private static boolean ifTableNotExists(@Nullable Throwable e) {
        var cause = e;
        while (cause != null) {
            if (cause instanceof UnexpectedResultException) {
                var unexpected = ((UnexpectedResultException) cause);
                if (unexpected.getStatusCode() != StatusCode.SCHEME_ERROR) {
                    return false;
                }

                for (var issue : ((UnexpectedResultException) cause).getIssues()) {
                    return issue.toString().contains("Table not found");
                }
            }

            cause = e.getCause();
        }
        return false;
    }

    private StructValue toStruct(Record record) {
        return StructValue.of(Map.of(
            "hash", PrimitiveValue.uint32(record.labels.hashCode()),
            "labels", PrimitiveValue.utf8(record.labels),
            "shardId", PrimitiveValue.uint32(record.shardId),
            "localId", PrimitiveValue.uint64(record.localId),
            "createdSeconds", PrimitiveValue.uint64(nowSeconds()),
            "flags", PrimitiveValue.uint32(record.flags)
        ));
    }

    private long nowSeconds() {
        return TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis());
    }
}
