package ru.yandex.solomon.experiments.gordiychuk.recovery.stockpile;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import java.util.stream.LongStream;

import com.google.common.net.HostAndPort;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monlib.metrics.JvmThreads;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.tool.KikimrHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.tool.cfg.SolomonPorts;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;
import ru.yandex.stockpile.kikimrKv.ShardIdMapToLong;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileRecoveryTask implements AutoCloseable {
    private ShardIdMapToLong mapping;
    private ExecutorService executor;
    private final KikimrKvClient kvClient;
    private final StockpileShardReader reader;
    private final StockpileShardRecovery recovery;
    private final StockpileClient stockpile;

    public StockpileRecoveryTask(SolomonCluster source, SolomonCluster target) {
        this.executor = new ForkJoinPool(Runtime.getRuntime().availableProcessors() / 2);
        this.kvClient = KikimrHelper.createKvClient(source);
        this.reader = new StockpileShardReader(kvClient);
        this.stockpile = StockpileClients.create(
            target.hosts().stream()
                .map(host -> HostAndPort.fromParts(host, SolomonPorts.STOCKPILE_GRPC))
                .collect(Collectors.toList()),
            StockpileClientOptions.newBuilder(
                    DefaultClientOptions.newBuilder()
                        .setRequestTimeOut(5, TimeUnit.MINUTES)
                        .setRpcExecutor(executor)
                        .setResponseHandlerExecutorService(executor)
                        .setClientId("recovery")
                        .setMaxOutboundMessageSizeInBytes((int) DataSize.fromGigaBytes(100).toBytes())
                        .setMaxInboundMessageSizeInBytes((int) DataSize.fromMegaBytes(256).toBytes()))
                .setExpireClusterMetadata(30, TimeUnit.SECONDS)
                .setRetryStopStrategy(StopStrategies.stopAfterAttempt(1))
                .build());
        JvmThreads.addExecutorMetrics("recovery", executor, MetricRegistry.root());
        this.recovery = new StockpileShardRecovery(reader, stockpile, executor);
        this.mapping = new ShardIdMapToLong(kvClient.resolveKvTablets(source.getSolomonVolumePath()).join());
    }

    public CompletableFuture<Void> recoveryShard(int shardId) {
        long tabletId = mapping.get(shardId);
        System.out.println("ShardId " + shardId + " mapped on tabletId " + tabletId);
        return recoveryTablet(tabletId);
    }

    public CompletableFuture<Void> recoveryLocalShards() {
        long[] localTabletIds;
        try(var kvClient = KikimrHelper.createKvClient("localhost")) {
            LongOpenHashSet set = new LongOpenHashSet(mapping.getValues());
            localTabletIds = LongStream.of(kvClient.findTabletsOnLocalhost().join())
                .filter(set::contains)
                .toArray();
        }

        AtomicInteger cursor = new AtomicInteger();
        AsyncActorBody body = () -> {
            int index = cursor.getAndIncrement();
            if (index >= localTabletIds.length) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            String progress = String.format("%.2f%%", index * 100. / localTabletIds.length);
            System.out.println("Recovery at " + HostUtils.getShortName() + " progress: " + progress);
            return recoveryTablet(localTabletIds[index]);
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, executor, 1);
        return runner.start()
            .whenComplete((ignore, e) -> {
                System.out.println("Complete local shard recovery");
            });
    }

    private CompletableFuture<Void> recoveryTablet(long tabletId) {
        System.out.println("Start recovery tabletId " + tabletId);
        return recovery.run(tabletId)
            .whenComplete((ignore, e) -> {
                MetricRegistry.root().counter("stockpile.recovery.shards.count").inc();
                System.out.println("Complete recovery tabletId " + tabletId);
            });
    }

    @Override
    public void close() {
        kvClient.close();
        stockpile.close();
        executor.shutdownNow();
    }
}
