package ru.yandex.solomon.experiments.gordiychuk.recovery.stockpile;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Throwables;
import com.google.protobuf.ByteString;

import ru.yandex.kikimr.client.kv.KikimrKvClient;
import ru.yandex.kikimr.util.NameRange;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.serializer.ByteStringsStockpile;
import ru.yandex.solomon.experiments.gordiychuk.recovery.MappingRecord;
import ru.yandex.solomon.tool.stockpile.backup.BackupsHelper;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientCounting;
import ru.yandex.stockpile.kikimrKv.counting.KikimrKvClientMetrics;
import ru.yandex.stockpile.kikimrKv.counting.ReadClass;
import ru.yandex.stockpile.kikimrKv.counting.WriteClass;
import ru.yandex.stockpile.server.SnapshotLevel;
import ru.yandex.stockpile.server.data.index.SnapshotIndex;
import ru.yandex.stockpile.server.data.index.SnapshotIndexContentSerializer;
import ru.yandex.stockpile.server.data.names.FileNameParsed;
import ru.yandex.stockpile.server.data.names.FileNamePrefix;
import ru.yandex.stockpile.server.data.names.IndexAddressWithFileCount;
import ru.yandex.stockpile.server.data.names.file.IndexFile;
import ru.yandex.stockpile.server.shard.iter.SnapshotIterator;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardReader implements AutoCloseable {
    private static final long TS = Instant.parse("2019-01-01T00:00:00Z").toEpochMilli() / 1000;
    private static final FileNamePrefix.Backup BACKUP_PREFIX = new FileNamePrefix.Backup(TS);
    private KikimrKvClientCounting counting;
    private BackupsHelper backupsHelper;
    private volatile boolean closed;

    private final RetryConfig retryConfig = RetryConfig.DEFAULT
            .withNumRetries(Integer.MAX_VALUE)
            .withDelay(300)
            .withStats((timeSpentMillis, cause) -> {
                MetricRegistry.root().counter("stockpile.shard.read.retry").inc();
                System.out.println("Failed on read, retrying...\n" + Throwables.getStackTraceAsString(cause));
            })
            .withExceptionFilter(throwable -> !closed);

    public StockpileShardReader(KikimrKvClient kvClient) {
        this.counting = new KikimrKvClientCounting(kvClient, new KikimrKvClientMetrics(MetricRegistry.root()));
        this.backupsHelper = new BackupsHelper(kvClient);
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return RetryCompletableFuture.runWithRetries(supplier, retryConfig);
    }

    public CompletableFuture<Void> makeBackup(long tabletId) {
        return retry(() -> dropBackup(tabletId)
            .thenCompose(unit -> backupsHelper.backupCurrent(tabletId, 0, BACKUP_PREFIX)))
            .thenRun(() -> {
                System.out.println("Make backup for tabletId " + tabletId);
            });
    }

    public CompletableFuture<Void> dropBackup(long tabletId) {
        return retry(() -> backupsHelper.deleteBackupsAsync(tabletId, 0, BACKUP_PREFIX))
            .thenRun(() -> {
                System.out.println("Drop backup for tabletId " + tabletId);
            });
    }

    public CompletableFuture<SnapshotIterator> metricsIterator(long tabletId) {
        return metricsIterator(tabletId, SnapshotLevel.ETERNITY, false);
    }

    public CompletableFuture<SnapshotIterator> metricsIterator(long tabletId, SnapshotLevel level, boolean last) {
        return readIndex(tabletId, level, last)
                .thenApply(index -> {
                    var chunkIterator = new StockpileSnapshotChunkIterator(tabletId, BACKUP_PREFIX, index, counting, retryConfig);
                    return new SnapshotIterator(index, chunkIterator);
                });
    }

    private CompletableFuture<SnapshotIndex> readIndex(long tabletId, SnapshotLevel level, boolean last) {
        return retry(() -> backupsHelper.listFilesInBackupAsync(tabletId, 0, BACKUP_PREFIX))
            .thenCompose(files -> {
                var indexes = IndexAddressWithFileCount.fold(files.stream()
                    .map(file -> FileNameParsed.parseInBackup(BACKUP_PREFIX, file.getName()))
                    .filter(file -> file instanceof IndexFile)
                    .map(IndexFile.class::cast)
                    .filter(index -> index.getLevel() == level)
                    .collect(Collectors.toList()));

                var targetIndex = last ? indexes.get(indexes.size() - 1) : indexes.get(0);
                return Stream.of(targetIndex.indexFileNames())
                    .map(file -> retry(() -> counting.readDataLarge(ReadClass.START_READ_INDEX, tabletId, 0, file.reconstruct(BACKUP_PREFIX))))
                    .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                    .thenApply(chunks -> {
                        ByteString[] bytes = chunks.stream()
                            .map(ByteStringsStockpile::unsafeWrap)
                            .toArray(ByteString[]::new);

                        var content = SnapshotIndexContentSerializer.S.deserializeParts(bytes);
                        return new SnapshotIndex(targetIndex.getLevel(), targetIndex.getTxn(), content);
                    });
            })
            .whenComplete((ignore, e) -> {
                if (e != null) {
                    System.out.println("Read index for tabletId " + tabletId);
                }
            });
    }

    public CompletableFuture<List<MappingRecord>> readMapping(long tabletId) {
        return retry(() -> counting.readRangeNames(ReadClass.OTHER, tabletId, 0, NameRange.single("mapping")))
            .thenCompose(files -> {
                if (files.isEmpty()) {
                    return CompletableFuture.completedFuture(new byte[0]);
                }

                return retry(() -> counting.readDataLarge(ReadClass.OTHER, tabletId, 0, "mapping"));
            })
            .thenApply(bytes -> {
                if (bytes.length == 0) {
                    return List.of();
                }

                var result = new String(bytes)
                    .lines()
                    .map(MappingRecord::parse)
                    .sorted((o1, o2) -> Long.compareUnsigned(o1.localLocalId, o2.localLocalId))
                    .collect(toList());
                System.out.println("Read mapping for tabletId " + tabletId + ", records "+ DataSize.shortString(result.size()));
                return result;
            });
    }

    public CompletableFuture<Void> deleteMapping(long tabletId) {
        return retry(() -> counting.deleteRange(WriteClass.OTHER, tabletId, 0, NameRange.single("mapping")))
            .thenRun(() -> {
                System.out.println("Drop mapping for tabletId " + tabletId);
            });
    }

    @Override
    public void close() {
        closed = true;
    }
}
