package ru.yandex.solomon.experiments.gordiychuk.recovery.stockpile;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeDouble;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.archive.MetricArchiveImmutable;
import ru.yandex.solomon.experiments.gordiychuk.recovery.MappingRecord;
import ru.yandex.solomon.tool.stockpile.StockpileShardWriters;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.server.shard.iter.SnapshotIterator;
import ru.yandex.stockpile.server.shard.load.Async;

/**
 * @author Vladimir Gordiychuk
 */
public class StockpileShardRecovery {
    private final StockpileShardReader reader;
    private final StockpileClient stockpile;
    private final ExecutorService executor;
    private final Metrics read = new Metrics("stockpile.read");
    private final Metrics merge = new Metrics("stockpile.merge");
    private final Metrics skip = new Metrics("stockpile.ignore");

    public StockpileShardRecovery(StockpileShardReader reader, StockpileClient stockpile, ExecutorService executor) {
        this.reader = reader;
        this.stockpile = stockpile;
        this.executor = executor;
    }

    public CompletableFuture<Void> run(long tabletId) {
        return reader.readMapping(tabletId)
            .thenCompose(mapping -> {
                if (mapping.isEmpty()) {
                    System.out.println("Mapping at tabletId " + tabletId + " absent");
                    return CompletableFuture.completedFuture(null);
                }

                var writer = new StockpileShardWriters(stockpile, executor);
                return reader.makeBackup(tabletId)
                    .thenCompose(ignore -> reader.metricsIterator(tabletId))
                    .thenCompose(it -> merge(mapping, it, writer, tabletId))
                    .thenCompose(ignore -> {
                        System.out.println("Complete merge for tabletId " + tabletId);
                        writer.complete();
                        return writer.doneFuture();
                    })
                    .thenCompose(ignore -> reader.dropBackup(tabletId))
                    .thenCompose(ignore -> reader.deleteMapping(tabletId));
            })
            .whenComplete((ignore, e) -> System.out.println("Complete recovery for tabletId " + tabletId));
    }

    private CompletableFuture<Void> merge(List<MappingRecord> mappings, SnapshotIterator it, StockpileShardWriters writer, long tabletId) {
        AtomicInteger cursor = new AtomicInteger();
        GaugeDouble progressMetric =
            MetricRegistry.root().gaugeDouble("stockpile.recovery.shard.progress", Labels.of("tabletId", "" + tabletId));
        return Async.forEachAsync(it, metric -> {
            read.add(metric.archive());
            int pos = cursor.get();

            double progress = pos * 100. / mappings.size();
            progressMetric.set(progress);

            while (pos < mappings.size()) {
                var mapping = mappings.get(pos);
                int compare = Long.compareUnsigned(mapping.localLocalId, metric.localId());
                if (compare > 0) {
                    break;
                } else if (compare < 0) {
                    pos = cursor.incrementAndGet();
                } else {
                    if ((pos % 10_000) == 0) {
                        System.out.println("Recovery tabletId " + tabletId + "  progress: " + String.format("%.2f%%", progress));
                    }

                    cursor.incrementAndGet();
                    merge.add(metric.archive());
                    return writer.write(mapping.toMetric(metric.archive()));
                }
            }

            skip.add(metric.archive());
            return CompletableFuture.completedFuture(null);
        });
    }

    private static class Metrics {
        final Rate bytes;
        final Rate metrics;
        final Rate records;

        Metrics(String prefix) {
            var registry = MetricRegistry.root();
            bytes = registry.rate(prefix + ".bytes");
            metrics = registry.rate(prefix + ".metrics");
            records = registry.rate(prefix + ".records");
        }

        void add(MetricArchiveImmutable archive) {
            bytes.add(archive.bytesCount());
            records.add(archive.getRecordCount());
            metrics.add(1);
        }
    }
}
