#!/bin/bash
set -uo pipefail

ITYPES_FILE=itypes.txt
if [ ! -f "$ITYPES_FILE" ]; then
    echo "$ITYPES_FILE file does not exist";
    exit 1;
fi

COMMON_SIGNALS_REGEX="cpu-|cgroup-|disk-|fdstat-|instances-|iostat-|loadavg-|mem-|netstat-|sockstat-|hbf4-|hbf6-"

RESULT_PATH=result
TMP_PATH=tmp

# cleanup
rm -rf result
rm -rf tmp

# alerts
ALERTS_RESULT_PATH="$RESULT_PATH/alerts"
echo "checking yasm alerts..."
mkdir -p $ALERTS_RESULT_PATH

cat $ITYPES_FILE | while read itype; 
do
  curl -s "yasm.yandex-team.ru/srvambry/alerts/list?itype=${itype}" | \
  jq '.response.result[].signal' | \
  grep -E "$COMMON_SIGNALS_REGEX" | \
  sort -u \
  1>"$ALERTS_RESULT_PATH/${itype}.txt"
done 


# panels
PANELS_RESULT_PATH="$RESULT_PATH/panels"
echo
echo "checking yasm panels..."
mkdir -p $PANELS_RESULT_PATH
mkdir -p $TMP_PATH

PANEL_IDS_FILE="$TMP_PATH/panel_ids.txt"
curl -s "yasm.yandex-team.ru/srvambry/get_all?fields=_id&type=panel" | jq -r  ".result[]._id" 1>$PANEL_IDS_FILE
echo "`cat $PANEL_IDS_FILE | wc -l` panel ids found"

PANELS_TMP_PATH="$TMP_PATH/panels"
mkdir -p $PANELS_TMP_PATH
echo "loading panels json..."
cat $PANEL_IDS_FILE | while read panel_id; 
do
  curl -s "yasm.yandex-team.ru/srvambry/get_all?_id=$panel_id" \
  1>"$PANELS_TMP_PATH/${panel_id}.json" | echo "loaded $panel_id" &
done

BAD_PANELS="$TMP_PATH/bad_panels.txt"
rm -f $BAD_PANELS

ITYPE_TMP_PATH="$TMP_PATH/itype"
rm -rf $ITYPE_TMP_PATH
mkdir -p $ITYPE_TMP_PATH

while read itype; do
  ITYPES_TAG_REGEX=$(echo -n "itype=${itype}|${itype}_");
  ITYPES_SIGNAL_REGEX=$(echo -n "$itype=${itype}|:${itype}_");
	
  echo "itypes tagregex: $ITYPES_TAG_REGEX";
  echo "itypes signal regex: $ITYPES_SIGNAL_REGEX";

  ls $PANELS_TMP_PATH | while read panel_file;
  do
    #echo "panel file: $panel_file"

    # try first format  
    signals=$(
      jq \
          --arg signal_regex "${ITYPES_SIGNAL_REGEX}" \
          --arg tag_regex "${ITYPES_TAG_REGEX}" \
          '.result[].charts[].signals[]? | (select(type == "string") | select(.|test($signal_regex)) // {}) // (select(.signal != null) | select(.signal|test($signal_regex)) | .signal) // (select(.tag != null) | select(.tag|test($tag_regex)) | .name)' \
          "$PANELS_TMP_PATH/$panel_file" 2>/dev/null;
    );

    if [ $? -eq 0 ]
    then
      echo "$panel_file: type 1";
    else
      # try second format
      signals=$(
        jq \
          --arg signal_regex "${ITYPES_SIGNAL_REGEX}" \
          --arg tag_regex "${ITYPES_TAG_REGEX}" \
          '.result[].value.charts[].signals[]? | (select(type == "string") | select(.|test($signal_regex)) // {}) // (select(.signal != null) | select(.signal|test($signal_regex)) | .signal) // (select(.tag != null) | select(.tag|test($tag_regex)) | .name)' \
          "$PANELS_TMP_PATH/$panel_file" 2>/dev/null
      );

      if [ $? -eq 0 ]
      then
        echo "$panel_file: type 2";
      else
        # try third format
        signals=$(
          jq \
            --arg signal_regex "${ITYPES_SIGNAL_REGEX}" \
            --arg tag_regex "${ITYPES_TAG_REGEX}" \
            '.result[].value[].signals[]? | (select(type == "string") | select(.|test($signal_regex)) // {}) // (select(.signal != null) | select(.signal|test($signal_regex)) | .signal) // (select(.tag != null) | select(.tag|test($tag_regex)) | .name)' \
            "$PANELS_TMP_PATH/$panel_file" 2>/dev/null
        );

        if [ $? -eq 0 ]
        then
          echo "$panel_file: type 3";
        else
          echo "$panel_file: bad";
          echo $panel_file >> $BAD_PANELS;
        fi
      fi
    fi

    echo $signals | \
    grep -E "$COMMON_SIGNALS_REGEX" | \
    sort -u \
    1>>"$ITYPE_TMP_PATH/${itype}.txt";
  done;

  cat "$ITYPE_TMP_PATH/${itype}.txt" | \
  sort -u \
  1>"$PANELS_RESULT_PATH/${itype}.txt"  
done < $ITYPES_FILE

# results
echo
echo "see result in `pwd`/$RESULT_PATH"
echo
echo 'examples:'
echo 'cat $ALERTS_RESULT_PATH/int.txt | grep -E "cpu-|cgroup-|disk-|fdstat-|instances-|iostat-|loadavg-|mem-|netstat-|sockstat-|hbf4-|hbf6-"'
echo "cat $ALERTS_RESULT_PATH/int.txt | grep -v portoinst"
echo 'cat $PANELS_RESULT_PATH/int.txt | grep -E "cpu-|cgroup-|disk-|fdstat-|instances-|iostat-|loadavg-|mem-|netstat-|sockstat-|hbf4-|hbf6-"'
echo "cat $PANELS_RESULT_PATH/int.txt | grep -v portoinst"

