package ru.yandex.solomon.experiments.iserezhik;

import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import ru.yandex.monitoring.api.v3.ChartWidget;
import ru.yandex.monitoring.api.v3.Dashboard;
import ru.yandex.monitoring.api.v3.Widget;
import ru.yandex.solomon.conf.db3.ydb.Entity;
import ru.yandex.solomon.conf.db3.ydb.YdbMonitoringDashboardsDao;
import ru.yandex.solomon.expression.PositionRange;
import ru.yandex.solomon.expression.SelParser;
import ru.yandex.solomon.expression.analytics.ProgramWithReturn;
import ru.yandex.solomon.expression.ast.Ast;
import ru.yandex.solomon.expression.ast.AstBinOp;
import ru.yandex.solomon.expression.ast.AstCall;
import ru.yandex.solomon.expression.ast.AstIdent;
import ru.yandex.solomon.expression.ast.AstInterpolatedString;
import ru.yandex.solomon.expression.ast.AstObject;
import ru.yandex.solomon.expression.ast.AstSelector;
import ru.yandex.solomon.expression.ast.AstSelectors;
import ru.yandex.solomon.expression.ast.AstStatement;
import ru.yandex.solomon.expression.ast.AstTernaryOp;
import ru.yandex.solomon.expression.ast.AstUnaryOp;
import ru.yandex.solomon.expression.ast.AstValue;
import ru.yandex.solomon.expression.ast.AstValueString;
import ru.yandex.solomon.expression.ast.RecursiveAstVisitor;
import ru.yandex.solomon.expression.ast.serialization.AstMappingContext;
import ru.yandex.solomon.expression.compile.CompileContext;
import ru.yandex.solomon.expression.exceptions.ParserException;
import ru.yandex.solomon.gateway.api.v3.intranet.dto.DashboardDtoConverter;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

public class UpdateDashboardsWithExtraWrapper {

    private static final AstVisitor visitor = new AstVisitor();
    private static final PositionRange U = PositionRange.UNKNOWN;
    private static final Pattern IS_SIGNAL_SUFFIX_VALID = Pattern.compile("([advehmntx]{4}|summ|hgram|max)");

    public static void main(String[] args) {
        var ydb = YdbHelper.createYdbClient(SolomonCluster.PROD_KFRONT);
        var dao = new YdbMonitoringDashboardsDao(ydb.table, SolomonCluster.PROD_KFRONT.kikimrRootPath() +
                "/Config/V3" +
                "/MonitoringDashboard", false);
        List<Entity> join = dao.readAll().join();
        List<Dashboard> receivedDashboards =
                join.stream().map(DashboardDtoConverter::fromEntity).collect(Collectors.toList());
        for (Dashboard receivedDashboard : receivedDashboards) {
            Dashboard updatedDashboard = updateDashboard(receivedDashboard);
            if (!receivedDashboard.equals(updatedDashboard)) {
//                dao.update(DashboardDtoConverter.toEntity(updatedDashboard,
//                        DashboardDtoConverter.parseVersion(receivedDashboard.getEtag())));
            }
        }
        ydb.close();
        System.exit(0);
    }

    private static Dashboard updateDashboard(Dashboard dashboard) {
        Dashboard.Builder dashboardBuilder = Dashboard.newBuilder(dashboard);
        List<Widget> widgetsList = dashboard.getWidgetsList();
        boolean isDashboardUpdated = false;
        for (int i = 0; i < widgetsList.size(); i++) {
            Widget oldWidget = widgetsList.get(i);
            ChartWidget.Queries query = updateQueries(oldWidget.getChart().getQueries());
            if (!oldWidget.getChart().getQueries().equals(query)) {
                isDashboardUpdated = true;
                ChartWidget chart = ChartWidget.newBuilder(oldWidget.getChart())
                        .setQueries(query)
                        .build();
                dashboardBuilder.setWidgets(i, Widget.newBuilder(oldWidget)
                        .setChart(chart)
                        .build());
            }
        }
        if (!isDashboardUpdated) {
            return dashboard;
        }
        return dashboardBuilder.build();
    }

    private static ChartWidget.Queries updateQueries(ChartWidget.Queries oldQueries) {
        var queriesBuilder = ChartWidget.Queries.newBuilder(oldQueries);
        List<ChartWidget.Queries.Target> oldTargetList = oldQueries.getTargetsList();
        boolean isTargetsUpdated = false;
        for (int i = 0; i < oldTargetList.size(); i++) {
            try {
                ChartWidget.Queries.Target newTarget = updateTarget(oldTargetList.get(i));
                if (!oldTargetList.get(i).equals(newTarget)) {
                    isTargetsUpdated = true;
                    queriesBuilder.setTargets(i, newTarget);
                }
            } catch (Exception ignore) {
            }
        }
        if (!isTargetsUpdated) {
            return oldQueries;
        }
        return queriesBuilder.build();
    }

    private static ChartWidget.Queries.Target updateTarget(ChartWidget.Queries.Target oldTarget) {
        var targetBuilder = ChartWidget.Queries.Target.newBuilder(oldTarget);
        System.out.println("---------------------------------------------------------");
        System.out.println("old query : " + oldTarget.getQuery());
        visitor.isAstUpdated = false;
        String oldQuery = oldTarget.getQuery();
        AstMappingContext mappingContext = new AstMappingContext(false);
        StringBuilder query = new StringBuilder();
        if (oldQuery.startsWith("//")) {
            query.append(oldQuery, 0, oldQuery.indexOf('\n') + 1);
        }
        Ast oldAst;
        try {
            oldAst = new SelParser(oldTarget.getQuery(), false).parseExpr();
        } catch (ParserException e) {
            ProgramWithReturn programWithReturn = new SelParser(oldTarget.getQuery()).parseProgramWithReturn();
            oldAst = programWithReturn.getExpr();
            for (AstStatement statement : programWithReturn.getStatements()) {
                query.append(mappingContext.renderToString(statement));
                query.append('\n');
            }
        }
        Ast newAst = visitor.visit(oldAst, null);
        if (!visitor.isAstUpdated()) {
            return oldTarget;
        }
        query.append(mappingContext.renderToString(newAst));
        System.out.println("new query : " + query);
        System.out.println("---------------------------------------------------------");
        return targetBuilder.setQuery(query.toString()).build();
    }


    private static Ast addOneMoreWrapToSelectors(AstSelectors selectors) {
        AstSelector signalSelector =
                selectors.getSelectors().stream()
                        .filter(selector ->
                                selector.getKey() instanceof AstValueString key &&
                                        key.getValue().equals("signal"))
                        .findAny().orElseThrow(UnsupportedOperationException::new);
        String signal = ((AstValueString) signalSelector.getValue()).getValue();
        String suffix = signal.substring(signal.lastIndexOf('_') + 1);
        String ident = switch (suffix.charAt(2)) {
            case 'r', 'h', 'm', 'e' -> "series_sum";
            case 'n' -> "series_min";
            case 'x' -> "series_max";
            case 'v' -> "series_avg";
            default -> "";
        };
        return new AstCall(U, new AstIdent(U, ident),
                List.of(new AstValueString(U, "signal"),
                        new AstSelectors(U, "", selectors.getSelectors())));
    }

    private static boolean isOneMoreWrapNeeded(AstSelectors selectors) {
        boolean result = false;
        boolean isYasmProject = selectors.getSelectors().stream().anyMatch(selector ->
                selector.getKey() instanceof AstValueString key &&
                        selector.getValue() instanceof AstValueString value &&
                        key.getValue().equals("project") &&
                        value.getValue().startsWith("yasm_"));
        boolean isSuffixValid = selectors.getSelectors().stream().anyMatch(selector ->
                selector.getKey() instanceof AstValueString key &&
                        selector.getValue() instanceof AstValueString value &&
                        key.getValue().equals("signal") &&
                        IS_SIGNAL_SUFFIX_VALID.matcher(value.getValue()).find()
        );
        if (isYasmProject && isSuffixValid) {
            for (AstSelector selector : selectors.getSelectors()) {
                if (
                        selector.getKey() instanceof AstValueString key &&
                                !key.getValue().equals("hosts") &&
                                !key.getValue().equals("signal") &&
                                isSelectorNotExact(selector)
                ) {
                    return selectors.getSelectors().stream().anyMatch(astSelector ->
                            astSelector.getKey() instanceof AstValueString streamKey &&
                                    streamKey.getValue().equals("signal"));

                }
            }
        }
        return result;
    }

    private static boolean isSelectorNotExact(AstSelector selector) {
        Ast key = selector.getKey();
        Ast value = selector.getValue();
        if (!(key instanceof AstValueString) ||
                !(value instanceof AstValueString)) {
            return true;
        }
        return !selector.getType()
                .create(
                        ((AstValueString) key).getValue(),
                        ((AstValueString) value).getValue()
                ).isExact();
    }

    static class AstVisitor extends RecursiveAstVisitor<CompileContext, Ast> {
        private boolean isAstUpdated;


        public boolean isAstUpdated() {
            return isAstUpdated;
        }

        @Override
        protected Ast visitIdent(AstIdent ast, @Nullable CompileContext context) {
            return ast;
        }

        @Override
        protected Ast visitInterpolatedString(AstInterpolatedString ast, @Nullable CompileContext context) {
            return ast;
        }

        @Override
        protected Ast visitBinOp(AstBinOp ast, @Nullable CompileContext context) {
            var newLeft = visit(ast.getLeft(), context);
            var newRight = visit(ast.getRight(), context);
            return new AstBinOp(ast.getRange(), newLeft, newRight, ast.getOp());
        }

        @Override
        protected Ast visitTernaryOp(AstTernaryOp ast, @Nullable CompileContext context) {
            var newLeft = visit(ast.getLeft(), context);
            var newRight = visit(ast.getRight(), context);
            return new AstTernaryOp(ast.getRange(), newLeft, newRight, ast.getCondition());
        }

        @Override
        protected Ast visitUnaryOp(AstUnaryOp ast, CompileContext context) {
            var newParam = visit(ast.getParam(), context);
            return new AstUnaryOp(ast.getRange(), newParam, ast.getOp());
        }

        @Override
        protected Ast visitCall(AstCall ast, @Nullable CompileContext context) {
            if (ast.func.getIdent().startsWith("series_")) {
                return ast;
            }
            List<Ast> oldArgs = ast.args;
            List<Ast> newArgs = oldArgs.stream()
                    .map(astArg -> visit(astArg, context))
                    .collect(Collectors.toList());
            if (oldArgs.equals(newArgs)) {
                return ast;
            }
            return new AstCall(ast.getRange(), ast.func, newArgs);
        }

        @Override
        protected Ast visitValue(AstValue ast, @Nullable CompileContext context) {
            return ast;
        }

        @Override
        protected Ast visitSelectors(AstSelectors ast, @Nullable CompileContext context) {
            var isWrapNeeded = isOneMoreWrapNeeded(ast);
            if (!isWrapNeeded) {
                return ast;
            }
            isAstUpdated = true;
            return addOneMoreWrapToSelectors(ast);
        }

        @Override
        protected Ast visitObject(AstObject ast, @Nullable CompileContext context) {
            return ast;
        }
    }
}

