package ru.yandex.solomon.experiments.jamel;

import java.util.concurrent.ForkJoinPool;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbClustersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServicesDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Sergey Polovko
 */
public class UpdateYasmShardSettings implements AutoCloseable {

    private static final int GRID_SECONDS = 10;

    private final YdbClient ydb;
    private final ServicesDao serviceDao;
    private final ShardsDao shardDao;
    private final ClustersDao clustersDao;

    public UpdateYasmShardSettings(SolomonCluster cluster) {
        this.ydb = YdbHelper.createYdbClient(cluster);
        var mapper = new ObjectMapper();
        serviceDao = new YdbServicesDao(ydb.table, cluster.kikimrRootPath() + "/Config/V2/Service", mapper, ForkJoinPool.commonPool());
        shardDao = new YdbShardsDao(ydb.table, cluster.kikimrRootPath() + "/Config/V2/Shard", mapper, ForkJoinPool.commonPool());
        clustersDao = new YdbClustersDao(ydb.table, cluster.kikimrRootPath() + "/Config/V2/Cluster", mapper, ForkJoinPool.commonPool());
    }

    public static void main(String[] args) {
        try (var cli = new UpdateYasmShardSettings(SolomonCluster.PRESTABLE_FRONT)) {
            cli.updateShardSettings();
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
        System.exit(0);
    }

    private void updateShardSettings() {
        {
            var services = serviceDao.findAll().join();
            System.out.printf("Processing %d services\n", services.size());
            int left = services.size();
            for (Service service : services) {
                System.out.printf("\r    left %d", left--);
                if (!isYasmStsProject(service.getProjectId())) {
                    continue;
                }

                if (service.getShardSettings().getGrid() == GRID_SECONDS) {
                    continue;
                }

                while (serviceDao.partialUpdate(patchService(service)).join().isEmpty()) {
                    var getOptional =
                        serviceDao.findOne(service.getProjectId(), service.getFolderId(), service.getId()).join();
                    if (getOptional.isPresent()) {
                        service = getOptional.get();
                    } else {
                        break;
                    }
                }
            }
        }

        {
            var clusters = clustersDao.findAll().join();
            System.out.printf("\nProcessing %d clusters\n", clusters.size());
            int left = clusters.size();
            for (Cluster cluster : clusters) {
                System.out.printf("\r    left %d", left--);
                if (!isYasmStsProject(cluster.getProjectId())) {
                    continue;
                }

                if (cluster.getShardSettings().getGrid() == 0) {
                    continue;
                }

                while (clustersDao.partialUpdate(patchCluster(cluster)).join().isEmpty()) {
                    var getOptional =
                        clustersDao.findOne(cluster.getProjectId(), "", cluster.getId()).join();
                    if (getOptional.isPresent()) {
                        cluster = getOptional.get();
                    } else {
                        break;
                    }
                }
            }
        }

        {
            var shards = shardDao.findAll().join();
            System.out.printf("\nProcessing %d shards\n", shards.size());
            int left = shards.size();
            for (Shard shard : shards) {
                System.out.printf("\r    left %d", left--);
                if (!isYasmStsProject(shard.getProjectId())) {
                    continue;
                }

                if (shard.getShardSettings().getGrid() == 0) {
                    continue;
                }

                while (shardDao.partialUpdate(patchShard(shard), false).join().isEmpty()) {
                    var getOptional = shardDao.findOne(shard.getProjectId(), shard.getFolderId(), shard.getId()).join();
                    if (getOptional.isPresent()) {
                        shard = getOptional.get();
                    } else {
                        break;
                    }
                }
            }
        }
    }

    private static ShardSettings patchGrid(ShardSettings s, int newGridSeconds) {
        return ShardSettings.of(
            s.getType(),
            s.getPullSettings(),
            newGridSeconds,
            s.getMetricsTtl(),
            s.getRetentionPolicy(),
            s.getAggregationSettings(),
            s.getInterval());
    }

    private static Service patchService(Service service) {
        return service.toBuilder()
            .setShardSettings(patchGrid(service.getShardSettings(), GRID_SECONDS))
            .build();
    }

    private static Cluster patchCluster(Cluster cluster) {
        // disable grid
        return cluster.toBuilder()
            .setShardSettings(patchGrid(cluster.getShardSettings(), 0))
            .build();
    }

    private static Shard patchShard(Shard shard) {
        // disable grid
        return shard.toBuilder()
            .setShardSettings(patchGrid(shard.getShardSettings(), 0))
            .build();
    }

    private static boolean isYasmStsProject(String projectId) {
        return projectId.startsWith("yasm_pull_sts");
    }

    @Override
    public void close() {
        ydb.close();
    }
}
