package ru.yandex.solomon.experiments.prudent;

import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.dao.ydb.entity.YdbAlertsDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.domain.template.AlertParameter;
import ru.yandex.solomon.alert.template.domain.AlertTemplateParameter;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;

/**
 * @author Nuradil Zhambyl
 */
public class ClustersCategorizer {

    private static final String ALL_CLUSTERS_FILE = "/Users/prudent/arcadia/solomon/misc/experiments/src/prudent/allClusters.txt";
    private static final String CLUSTERS_ALERTED_MONTH_MORE = "/Users/prudent/arcadia/solomon/misc/experiments/src/prudent/clustersAlertedMonthMore.txt";
    private static final String ALERTLESS_CLUSTERS = "/Users/prudent/arcadia/solomon/misc/experiments/src/prudent/clustersAlertless.txt";
    private static final SolomonCluster CLUSTER = SolomonCluster.PROD_KFRONT;

    private static final long DAY = 24 * 3600L * 1000;
    private static final long MONTH = 30 * DAY;

    private static Set<String> readFile(String fileName) throws IOException {
        Stream<String> stream = Files.lines(Paths.get(fileName));
        return stream.collect(Collectors.toSet());
    }

    private static void writeToFile(Set<String> set, String fileName) throws IOException {
        try (FileWriter fileWriter = new FileWriter(fileName);
             BufferedWriter writer = new BufferedWriter(fileWriter)) {
            boolean first = true;
            for (String cluster : set) {
                if (!first) {
                    writer.append('\n');
                }
                writer.append(cluster);
                first = false;
            }
        }
        System.out.printf("wrote to %s\n", fileName);
    }

    public static void main(String[] args) throws IOException {
        Set<String> allClusters = readFile(ALL_CLUSTERS_FILE);
        YdbClient ydb = YdbHelper.createYdbClient(CLUSTER);
        var mapper = new ObjectMapper();
        YdbAlertsDao ydbAlertsDao = new YdbAlertsDao(CLUSTER.kikimrRootPath(), ydb.table, ydb.scheme, YdbSchemaVersion.CURRENT, mapper);
        var alerts = ydbAlertsDao.listAll().join();
        Set<String> clustersAlertedMonthMore = new HashSet<>();
        long now = System.currentTimeMillis();
        Set<String> managedPostgres = new HashSet<>();
        for (Alert alert : alerts) {
            if (alert instanceof AlertFromTemplatePersistent alertFromTemplate) {
                if (alertFromTemplate.getTemplateId().startsWith("managed-postgres")) {
                    var paramsList = alertFromTemplate.getParameters();
                    for (var param : paramsList) {
                        String paramName = param.getName();
                        if ("cluster".equals(paramName) && param.getType().equals(AlertTemplateParameter.ParameterValueType.TEXT)) {
                            AlertParameter.TextParameterValue clusterParam = (AlertParameter.TextParameterValue) param;
                            String clusterName = clusterParam.getValue();
                            managedPostgres.add(clusterName);
                            if (now - alertFromTemplate.getCreatedAt() > MONTH && !clusterName.isEmpty()) {
                                clustersAlertedMonthMore.add(clusterName);
                            }
                        }
                    }
                }
            }
        }
        Set<String> alertlessClusters = new HashSet<>();
        for (String cluster : allClusters) {
            if (!managedPostgres.contains(cluster)) {
                alertlessClusters.add(cluster);
            }
        }
        writeToFile(clustersAlertedMonthMore, CLUSTERS_ALERTED_MONTH_MORE);
        writeToFile(alertlessClusters, ALERTLESS_CLUSTERS);
        System.exit(0);
    }
}
