package ru.yandex.solomon.experiments.prudent;


import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.grpc.Status;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.discovery.cluster.ClusterMapperImpl;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertsDao;
import ru.yandex.solomon.alert.dao.ydb.YdbAlertTemplateDao;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.dao.ydb.entity.YdbAlertsDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.config.SolomonConfigs;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayConfig;
import ru.yandex.solomon.config.thread.StubThreadPoolProvider;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.gateway.data.DataClient;
import ru.yandex.solomon.gateway.data.DataClientMetrics;
import ru.yandex.solomon.gateway.data.DataRequest;
import ru.yandex.solomon.gateway.data.DataResponse;
import ru.yandex.solomon.gateway.data.DownsamplingOptions;
import ru.yandex.solomon.gateway.data.DownsamplingType;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.MetricsClientFactory;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayListOrView;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.time.Interval;

import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;
import static ru.yandex.solomon.util.future.RetryCompletableFuture.runWithRetries;

/**
 * @author Nuradil Zhambyl
 */
public class ExportAlertComputationsFromTemplates implements AutoCloseable {

    private final YdbClient ydb;
    private final String ytTable;
    private final String ytAddress;
    private final AlertsDao alertsDao;
    private final AlertTemplateDao alertTemplateDao;
    private final DataClient dataClient;
    private final String installation;

    private static final long DAY = 24 * 3600L * 1000;
    private static final long WEEK = 7 * DAY;
    private static final long MONTH = 30 * DAY;

    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withExceptionFilter(throwable -> {
                Status status = Status.fromThrowable(throwable);
                return status.getCode() == Status.Code.UNAVAILABLE;
            })
            .withNumRetries(5)
            .withDelay(1_000)
            .withMaxDelay(60_000);

    private <T> CompletableFuture<T> run(Supplier<CompletableFuture<T>> taskProducer) {
        return runWithRetries(() -> safeCall(taskProducer::get), RETRY_CONFIG);
    }

    public ExportAlertComputationsFromTemplates(SolomonCluster cluster, String ytTable, String yt, String gatewayConfig, String installation) {
        this.ydb = YdbHelper.createYdbClient(cluster);
        this.ytTable = ytTable;
        this.ytAddress = yt;
        this.installation = installation;
        var mapper = new ObjectMapper();
        alertsDao = new YdbAlertsDao(cluster.kikimrRootPath(), ydb.table, ydb.scheme, YdbSchemaVersion.CURRENT, mapper);
        alertTemplateDao = new YdbAlertTemplateDao(cluster.kikimrRootPath(), ydb.table, ydb.scheme, YdbSchemaVersion.CURRENT, mapper);
        TGatewayConfig config = SolomonConfigs.parseConfig(gatewayConfig, TGatewayConfig.getDefaultInstance());
        ThreadPoolProvider threadPool = new StubThreadPoolProvider();
        ClusterMapper clusterMapper = new ClusterMapperImpl(config.getClustersConfigList(), DiscoveryService.async(), threadPool.getExecutorService("", ""), threadPool.getSchedulerExecutorService());
        MetricRegistry registry = new MetricRegistry();
        SolomonConfHolder confHolder = new SolomonConfHolder();
        confHolder.onConfigurationLoad(SolomonConfWithContext.EMPTY);
        ClientOptionsFactory clientOptionsFactory = new ClientOptionsFactory(Optional.empty(), Optional.empty(), threadPool);
        var metricsClientFactory = new MetricsClientFactory(threadPool, registry, clusterMapper, clientOptionsFactory);
        var flagsHolder = new FeatureFlagHolderStub();
        MetricsClient metricsClient = metricsClientFactory.create(
                "expression-gateway",
                config.getMetabaseClientConfig(),
                config.getStockpileClientConfig(),
                config.getDataProxyClientConfig(),
                flagsHolder);
        dataClient = new DataClient(new DataClientMetrics(), metricsClient, clusterMapper, registry, confHolder, flagsHolder);
    }

    private static long dateToMillis(String dateString) throws ParseException {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        Date date = formatter.parse(dateString);
        return date.getTime();
    }

    private void exportYtTable(Set<String> templateIds, String fromDate, String toDate) throws ParseException {
        long until, upto;
        try {
            until = dateToMillis(fromDate);
            upto = dateToMillis(toDate);
        } catch(ParseException e) {
            System.err.println("Dates not in yyyy-MM-dd format");
            throw e;
        }
        exportYtTable(templateIds, until, upto);
    }

    private void exportYtTable(Set<String> templateIds, long until, long upto) {
        List<YTreeMapNode> entries = new ArrayList<>(60_000_000);
        long step = WEEK;
        var alertTemplatesList = alertTemplateDao.getAll().join();
        Set<String> multiAlertIds = new HashSet<>();
        for (var alertTemplate : alertTemplatesList) {
            if (!alertTemplate.getGroupByLabels().isEmpty()) {
                multiAlertIds.add(alertTemplate.getId());
            }
        }
        Yt yt = YtUtils.http(ytAddress);
        Map<String, YPath> tableByTemplateId = new HashMap<>();
        for (String templateId : templateIds) {
            tableByTemplateId.put(templateId, YPath.simple(ytTable + "/" + templateId).append(true));
        }
        YdbAlertsDao ydbAlertsDao = (YdbAlertsDao) alertsDao;
        var alerts = ydbAlertsDao.listAll().join();
        int alertCount = 0;
        int totalEntriesCount = 0;
        for (Alert alert : alerts) {
            var alertId = alert.getId();
            if (alert instanceof AlertFromTemplatePersistent alertFromTemplate) {
                var templateId = alertFromTemplate.getTemplateId();
                if (!templateIds.contains(templateId)) {
                    continue;
                }
                String projectId = alertFromTemplate.getProjectId();
                System.out.printf("processing projectId: %s, templateId: %s\n", projectId, templateId);
                long now = upto;
                DataResponse response;
                boolean isMultiAlert = multiAlertIds.contains(templateId);
                String program;
                if (isMultiAlert) {
                    program = String.format("{cluster=\"%s\", service=\"alerting_statuses\", sensor=\"multiAlert.evaluation.status\", projectId=\"%s\", alertId=\"%s\", status!=\"*:MUTED\"}", installation, projectId, alertId);
                } else {
                    program = String.format("mod({cluster=\"%s\", service=\"alerting_statuses\", sensor=\"alert.evaluation.status\", projectId=\"%s\", alertId=\"%s\"}, 64)", installation, projectId, alertId);
                }
                do {
                    long from = Math.max(now - step, until);
                    DataRequest request = DataRequest
                            .newBuilder()
                            .setInterval(Interval.millis(from, now))
                            .setProgram(program)
                            .setDownsampling(DownsamplingOptions
                                    .newBuilder()
                                    .setDownsamplingType(DownsamplingType.OFF)
                                    .build())
                            .setProjectId("solomon")
                            .setDeadline(Instant.now().plusSeconds(60))
                            .build();
                    long finalNow = now;
                    response = run(() -> dataClient.readData(request)).exceptionally(e -> {
                        System.err.printf("could not read interval [%d, %d] of templateId %s in projectId %s", from, finalNow, templateId, projectId);
                        e.printStackTrace();
                        return null;
                    }).join();
                    if (response != null && !parseResponseToEntries(response, entries, alertId, isMultiAlert)) {
                        break;
                    }
                    var table = tableByTemplateId.get(templateId);
                    yt.tables().write(table, YTableEntryTypes.YSON, entries);
                    totalEntriesCount += entries.size();
                    entries.clear();
                    now -= step + 1;
                } while (now >= until);
                System.out.printf("processed projectId: %s, templateId: %s\n", projectId, templateId);
                alertCount++;
                System.out.println("alertCount: " + alertCount);
                System.out.println("entries: " + totalEntriesCount);
            }
        }
        System.out.println("Exported alert computations: " + totalEntriesCount);
    }

    private boolean parseResponseToEntries(
            DataResponse response,
            List<YTreeMapNode> entries,
            String alertId,
            boolean isMultiAlert)
    {
        int initialSize = entries.size();
        if (isMultiAlert) {
            var evalResult = response.getEvalResult();
            var vector = evalResult.castToVector();
            var array = vector.valueArray();
            for (var element : array) {
                var namedGraphData = element.castToGraphData().getNamedGraphData();
                var aggrGraphDataArrayList = namedGraphData.getAggrGraphDataArrayList();
                var labels = namedGraphData.getLabels();
                String status = Objects.requireNonNull(labels.findByKey("status")).getValue();
                addEntries(aggrGraphDataArrayList, alertId, status, entries);
            }
        } else {
            var graphData = response.getEvalResult().castToGraphData();
            var namedGraphData = graphData.getNamedGraphData();
            var aggrGraphDataArrayList = namedGraphData.getAggrGraphDataArrayList();
            addEntries(aggrGraphDataArrayList, alertId, "", entries);
        }
        return entries.size() > initialSize;
    }

    private void addEntries(
            AggrGraphDataArrayListOrView list,
            String alertId,
            String potentialStatus,
            List<YTreeMapNode> entries)
    {
        for (int i = 0; i < list.length(); i++) {
            long tsMillis = list.getTsMillis(i);
            String date = Instant.ofEpochMilli(tsMillis).toString();
            int value = (int) list.getValueDivided(i);
            String status;
            int count;
            if (!potentialStatus.isBlank()) {
                status = potentialStatus;
                count = value;
            } else {
                var code = EvaluationStatus.Code.forNumber(value);
                status = code.toString();
                count = 1;
            }
            if (count > 0) {
                entries.add(createEntry(
                        alertId,
                        date,
                        status,
                        count,
                        tsMillis));
            }
        }
    }

    private YTreeMapNode createEntry(
            String alertId,
            String date,
            String status,
            int count,
            long tsMillis)
    {
        return YTree.mapBuilder()
                .key("alert_id").value(alertId)
                .key("date").value(date)
                .key("status").value(status)
                .key("count").value(count)
                .key("ts_millis").value(tsMillis)
                .buildMap();
    }

    private static void assertSize(String[] args, int size) throws IllegalArgumentException {
        if (args.length < size) {
            throw new IllegalArgumentException(String.format("arguments size < %d", size));
        }
    }

    private static void copyIfDate(String date, StringBuilder copy) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        try {
            formatter.parse(date);
            copy.append(date);
        } catch(ParseException ignore) {
            System.out.println("Warning: boundary not a date in yyyy-MM-dd format");
        }
    }

    private static long parseLongOrNegative(String number) {
        try {
            return Long.parseLong(number);
        } catch(NumberFormatException ignore) {
            return -1;
        }
    }

    // example program arguments: production //home/solomon/service_provider_alerts/alerts/stats/alerts_evaluation hahn.yt.yandex.net /Users/prudent/arcadia/solomon/configs/production/gateway.conf 2022-07-13 2022-07-14 managed-postgresql-cpu-usage managed-postgresql-master-wal-size-percent managed-postgresql-disk-free-bytes-percent
    public static void main(String[] args) throws IllegalArgumentException {
        assertSize(args, 6);
        String installation = args[0];
        var cluster = "production".equals(installation) ? SolomonCluster.PROD_KFRONT : SolomonCluster.PRESTABLE_FRONT;
        var table = args[1];
        var yt = args[2];
        String gatewayConfig = args[3];
        System.out.println("Start export alert computations from " + cluster + " to " + yt + table);
        long millisLeft = parseLongOrNegative(args[4]);
        long millisRight = parseLongOrNegative(args[5]);
        StringBuilder dateLeft = new StringBuilder();
        StringBuilder dateRight = new StringBuilder();
        copyIfDate(args[4], dateLeft);
        copyIfDate(args[5], dateRight);
        Set<String> templateIds = new HashSet<>(Arrays.asList(args).subList(6, args.length));
        try (var cli = new ExportAlertComputationsFromTemplates(cluster, table, yt, gatewayConfig, installation)) {
            if (Math.min(millisLeft, millisRight) >= 0) {
                cli.exportYtTable(templateIds, millisLeft, millisRight);
            } else if (!dateLeft.isEmpty() && !dateRight.isEmpty()) {
                cli.exportYtTable(templateIds, dateLeft.toString(), dateRight.toString());
            } else {
                throw new IllegalArgumentException("interval must be either two millis or dates of yyyy-MM-dd format");
            }
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
        System.out.println("Export finished");
        System.exit(0);
    }

    @Override
    public void close() {
        ydb.close();
    }
}
