package ru.yandex.solomon.experiments.prudent;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import io.grpc.Status;

import ru.yandex.discovery.DiscoveryServices;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.client.AlertingClients;
import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.ydb.YdbAlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.gateway.dto.alert.AlertDto;
import ru.yandex.solomon.alert.gateway.dto.alert.AlertFromTemplateDto;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse;
import ru.yandex.solomon.tool.YdbClient;
import ru.yandex.solomon.tool.YdbHelper;
import ru.yandex.solomon.tool.cfg.SolomonCluster;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryConfig;

import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;
import static ru.yandex.solomon.util.future.RetryCompletableFuture.runWithRetries;

/**
 * @author Nuradil Zhambyl
 */
public class ExportAlertlessClustersExplanations implements AutoCloseable {
    private final YdbClient ydb;
    private final String ytTable;
    private final String ytAddress;
    private final AlertApi alertApi;
    private final AlertTemplateLastVersionDao alertTemplateLastVersionDao;

    public ExportAlertlessClustersExplanations(SolomonCluster cluster, String ytTable, String yt, String alertingAddress) {
        this.ydb = YdbHelper.createYdbClient(cluster);
        this.ytTable = ytTable;
        this.ytAddress = yt;
        alertApi = AlertingClients.create(DiscoveryServices.resolve(Collections.singletonList(alertingAddress)));
        alertTemplateLastVersionDao = new YdbAlertTemplateLastVersionDao(cluster.kikimrRootPath(), ydb.table, ydb.scheme, YdbSchemaVersion.CURRENT);
    }

    private static void assertSize(String[] args, int size) throws IllegalArgumentException {
        if (args.length < size) {
            throw new IllegalArgumentException(String.format("arguments size < %d", size));
        }
    }

    private static void copyIfDate(String date, StringBuilder copy) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        try {
            formatter.parse(date);
            copy.append(date);
        } catch(ParseException ignore) {
            System.out.println("Warning: boundary not a date in yyyy-MM-dd format");
        }
    }

    private static long parseLongOrNegative(String number) {
        try {
            return Long.parseLong(number);
        } catch(NumberFormatException ignore) {
            return -1;
        }
    }

    private static long dateToMillis(String dateString) throws ParseException {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        Date date = formatter.parse(dateString);
        return date.getTime();
    }

    private void exportYtTable(Set<String> templateIds, String fromDate, String toDate, List<String> clusterNames, String postfix, int noDataCountLimit) throws ParseException {
        long until, upto;
        try {
            until = dateToMillis(fromDate);
            upto = dateToMillis(toDate);
        } catch(ParseException e) {
            System.err.println("Dates not in yyyy-MM-dd format");
            throw e;
        }
        exportYtTable(templateIds, until, upto, clusterNames, postfix, noDataCountLimit);
    }

    private void exportYtTable(Set<String> templateIds, long until, long upto, List<String> clusterNames, String postfix, int noDataCountLimit) {
        System.out.println( Instant.now() + " started");
        Yt yt = YtUtils.http(ytAddress);
        Map<String, YPath> tableByTemplateId = new HashMap<>();
        for (String templateId : templateIds) {
            tableByTemplateId.put(templateId, YPath.simple(ytTable + "/" + templateId + postfix).append(true));
        }
        Map<String, String> versionTagByTemplateId = new HashMap<>();
        var alertTemplateLastVersions = alertTemplateLastVersionDao.getAll().join();
        for (var alertTemplateLastVersion : alertTemplateLastVersions) {
            versionTagByTemplateId.put(alertTemplateLastVersion.id(), alertTemplateLastVersion.templateVersionTag());
        }
        List<ClusterTemplatePair> pairs = new ArrayList<>();
        for (String clusterName : clusterNames) {
            for (String templateId : templateIds) {
                pairs.add(new ClusterTemplatePair(clusterName, templateId));
            }
        }
        var actorBody = new ClusterExplainer(
                pairs,
                tableByTemplateId,
                yt,
                versionTagByTemplateId,
                until,
                upto,
                noDataCountLimit);
        var runner = new AsyncActorRunner(actorBody, ForkJoinPool.commonPool(), 50);
        runner.start().join();
        System.out.println( Instant.now() + " done");
    }

    private static boolean parseResponseToEntries(
            TExplainEvaluationResponse response,
            CopyOnWriteArrayList<YTreeMapNode> entries,
            String clusterName,
            long tsMillis) {
        String statusCode = response.getEvaluationStatus().getCode().toString();
        String date = Instant.ofEpochMilli(tsMillis).toString();
        entries.add(createEntry(
                clusterName,
                date,
                statusCode,
                tsMillis));
        return !response.getEvaluationStatus().getCode().equals(TEvaluationStatus.ECode.NO_DATA);
    }

    private static YTreeMapNode createEntry(
            String alertClusterId,
            String date,
            String status,
            long tsMillis)
    {
        return YTree.mapBuilder()
                .key("alert_id").value(alertClusterId)
                .key("date").value(date)
                .key("status").value(status)
                .key("count").value(1)
                .key("ts_millis").value(tsMillis)
                .buildMap();
    }

    private static AlertFromTemplateDto.TextParameterValue constructCluster(String value) {
        var cluster = new AlertFromTemplateDto.TextParameterValue();
        cluster.name = "cluster";
        cluster.value = value;
        return cluster;
    }

    private static AlertDto constructAlertDto(
            String templateId,
            String clusterName,
            String templateVersionTag) {
        AlertDto alertDto = new AlertDto();
        alertDto.projectId = "prudent_project_id";
        alertDto.type = new AlertDto.Type();
        alertDto.type.fromTemplate = new AlertFromTemplateDto();
        addFromTemplate(alertDto.type.fromTemplate, templateId, templateVersionTag, clusterName);
        alertDto.annotations = new HashMap<>();
        alertDto.description = "";
        alertDto.id = "";
        alertDto.name = templateId + clusterName;
        return alertDto;
    }

    private static void addFromTemplate(AlertFromTemplateDto fromTemplate, String templateId, String templateVersionTag, String clusterName) {
        fromTemplate.templateId = templateId;
        fromTemplate.templateVersionTag = templateVersionTag;
        fromTemplate.textValueParameters = new ArrayList<>();
        fromTemplate.textValueParameters.add(constructCluster(clusterName));
    }

    private static List<String> readFile(String fileName) throws IOException {
        Stream<String> stream = Files.lines(Paths.get(fileName));
        return stream.collect(Collectors.toList());
    }

    // example program arguments: production //home/solomon/service_provider_alerts/alerts/stats/alerts_evaluation hahn.yt.yandex.net 2022-06-22 2022-07-22 conductor_group://solomon_prod_alerting:8799 /Users/alextrushkin/arcadia/solomon/misc/experiments/src/prudent/clusterNames.txt _cluster_debug 100 managed-postgresql-disk-free-bytes-percent
    public static void main(String[] args) throws IllegalArgumentException, IOException {
        assertSize(args, 9);
        String installation = args[0];
        var cluster = "production".equals(installation) ? SolomonCluster.PROD_KFRONT : SolomonCluster.PRESTABLE_FRONT;
        var table = args[1];
        var yt = args[2];
        System.out.println("Start export explanations of alertless clusters from " + cluster + " to " + yt + table);
        long millisLeft = parseLongOrNegative(args[3]);
        long millisRight = parseLongOrNegative(args[4]);
        StringBuilder dateLeft = new StringBuilder();
        StringBuilder dateRight = new StringBuilder();
        copyIfDate(args[3], dateLeft);
        copyIfDate(args[4], dateRight);
        String alertingAddress = args[5];
        List<String> clusterNames = readFile(args[6]);
        String postfix = args[7];
        int noDataCountLimit = Integer.parseInt(args[8]);
        Set<String> templateIds = new HashSet<>(Arrays.asList(args).subList(9, args.length));
        try (var cli = new ExportAlertlessClustersExplanations(cluster, table, yt, alertingAddress)) {
            if (Math.min(millisLeft, millisRight) >= 0) {
                cli.exportYtTable(templateIds, millisLeft, millisRight, clusterNames, postfix, noDataCountLimit);
            } else if (!dateLeft.isEmpty() && !dateRight.isEmpty()) {
                cli.exportYtTable(templateIds, dateLeft.toString(), dateRight.toString(), clusterNames, postfix, noDataCountLimit);
            } else {
                throw new IllegalArgumentException("interval must be either two millis or dates of yyyy-MM-dd format");
            }
        } catch (Throwable t) {
            t.printStackTrace();
            System.exit(1);
        }
        System.out.println("Export finished");
        System.exit(0);
    }

    @Override
    public void close() {
        ydb.close();
    }

    private class ClusterExplainer implements AsyncActorBody {
        private final Iterator<ClusterTemplatePair> it;
        private final Map<String, YPath> tableByTemplateId;
        private final Yt yt;
        private final Map<String, String> versionTagByTemplateId;
        private final long until;
        private final long upto;
        private final int noDataCountLimit;
        private final AtomicInteger lostCount;

        private final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
                .withExceptionFilter(throwable -> {
                    Status status = Status.fromThrowable(throwable);
                    return status.getCode() == Status.Code.UNAVAILABLE || status.getCode() == Status.Code.DEADLINE_EXCEEDED;
                })
                .withNumRetries(5)
                .withDelay(1_000)
                .withMaxDelay(60_000);

        private <T> CompletableFuture<T> run(Supplier<CompletableFuture<T>> taskProducer) {
            return runWithRetries(() -> safeCall(taskProducer::get), RETRY_CONFIG);
        }

        public ClusterExplainer(
                List<ClusterTemplatePair> pairs,
                Map<String, YPath> tableByTemplateId,
                Yt yt,
                Map<String, String> versionTagByTemplateId,
                long until,
                long upto,
                int noDataCountLimit) {
            this.it = pairs.iterator();
            this.tableByTemplateId = tableByTemplateId;
            this.yt = yt;
            this.versionTagByTemplateId = versionTagByTemplateId;
            this.until = until;
            this.upto = upto;
            this.noDataCountLimit = noDataCountLimit;
            this.lostCount = new AtomicInteger(0);
        }

        @Override
        public CompletableFuture<?> run() {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }
            ClusterTemplatePair pair = it.next();
            String clusterName = pair.clusterName;
            String templateId = pair.templateId;
            System.out.printf("start {clusterName %s, templateId %s}\n", clusterName, templateId);
            return CompletableFuture.completedFuture(null).thenComposeAsync(unused -> {
                String templateVersionTag = versionTagByTemplateId.getOrDefault(templateId, null);
                long now = until;
                AtomicInteger noDataCount = new AtomicInteger();
                CopyOnWriteArrayList<YTreeMapNode> entries = new CopyOnWriteArrayList<>();
                CompletableFuture<Void> done = CompletableFuture.completedFuture(null);
                AlertDto alertDto = constructAlertDto(templateId, clusterName, templateVersionTag);
                while (now <= upto && noDataCount.get() < noDataCountLimit) {
                    TExplainEvaluationRequest request = TExplainEvaluationRequest.newBuilder()
                            .setAlert(alertDto.toProto())
                            .setEvaluationTimeMillis(now)
                            .build();
                    long finalNow = now;
                    done = done.thenCompose(ignore -> run(() -> alertApi.explainEvaluation(request.toBuilder()
                            .setDeadlineMillis(System.currentTimeMillis() + TimeUnit.SECONDS.toMillis(60))
                            .build())).exceptionally(e -> {
                        lostCount.getAndIncrement();
                        System.out.printf("lost %d until now\n", lostCount.get());
                        return null;
                    }).thenApply(response -> {
                        if (response != null) {
                            if (!parseResponseToEntries(response, entries, clusterName, finalNow)) {
                                noDataCount.getAndIncrement();
                            } else {
                                noDataCount.set(0);
                            }
                        }
                        return null;
                    }));
                    now += 30000;
                }
                System.out.printf("finish {clusterName %s, templateId %s}\n", clusterName, templateId);
                var table = tableByTemplateId.get(templateId);

                return done
                        .thenApply(unused1 -> {
                            yt.tables().write(table, YTableEntryTypes.YSON, entries);
                            return null;
                        });
            });
        }
    }

    private static record ClusterTemplatePair(String clusterName, String templateId) {
    }
}
