package ru.yandex.solomon.experiments.rorewillo.pusher;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.protobuf.SummaryDouble;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.client.ColumnFlagMask;

/**
 * @author Oleg Baryshnikov
 */
public class DoubleSummaryPusher {

    private static final Logger logger = LoggerFactory.getLogger(DoubleSummaryPusher.class);

    private static final long START_MILLIS = Instant.parse("2018-01-07T00:00:00Z").toEpochMilli();
    private static final long END_MILLIS = Instant.parse("2018-06-09T00:00:00Z").toEpochMilli();

    private static final int INTERVAL_BETWEEN_POINTS_MILLIS = 300_000;

    private static final Labels COMMON_LABELS = Labels.of(
        "project", "rorewillo-test",
        "cluster", "cluster",
        "service", "push_service",
        "version", "1");

    public static void main(String[] args) throws Exception {
        try (MetricPusher pusher = MetricPusher.create(logger)) {
            pusher.forceUpdate();
            pushSummary(pusher);
        }
    }

    private static void pushSummary(MetricPusher pusher) throws ExecutionException, InterruptedException {
        List<TPoint> points = generateSummaryPoints();
        saveAsPseudoSummary(pusher, points);
        saveAsSummary(pusher, points);
    }

    private static void saveAsSummary(MetricPusher pusher, List<TPoint> points)
        throws InterruptedException, ExecutionException
    {
        Labels labels = COMMON_LABELS.toBuilder().add("sensor", "dsummary_sensor").build();
        pusher.push(labels, MetricType.DSUMMARY, ColumnFlagMask.MINIMAL_DSUMMARY, points).get();
    }

    private static void saveAsPseudoSummary(MetricPusher pusher, List<TPoint> points)
        throws ExecutionException, InterruptedException
    {
        saveSummaryPart(pusher, points, SummaryDouble::getCount, "count");
        saveSummaryPart(pusher, points, SummaryDouble::getMax, "max");
        saveSummaryPart(pusher, points, SummaryDouble::getMin, "min");
        saveSummaryPart(pusher, points, SummaryDouble::getSum, "sum");
    }

    private static void saveSummaryPart(
        MetricPusher pusher,
        List<TPoint> points,
        Function<SummaryDouble, Number> func,
        String name) throws ExecutionException, InterruptedException
    {
        Labels labels = COMMON_LABELS.toBuilder()
            .add("sensor", "pseudo_dsummary_sensor")
            .add("aggrType", name)
            .build();

        List<TPoint> binPoints = points.stream()
            .map(hPoint -> {
                SummaryDouble summary = hPoint.getSummaryDouble();
                Number value = func.apply(summary);

                return TPoint.newBuilder()
                    .setDoubleValue(value.doubleValue())
                    .setTimestampsMillis(hPoint.getTimestampsMillis())
                    .build();
            })
            .collect(Collectors.toList());

        pusher.push(labels, MetricType.DGAUGE, ColumnFlagMask.MINIMAL_DOUBLE_MASK, binPoints).get();
    }

    private static List<TPoint> generateSummaryPoints() {
        List<TPoint> points = new ArrayList<>();

        Random random = new Random(System.nanoTime());

        long tsMillis = START_MILLIS;

        while (tsMillis < END_MILLIS) {
            int count = random.nextInt(100);

            int max = 0;
            int min = 0;
            int sum = 0;

            for (int i = 0; i < count; ++i) {
                int value = random.nextInt(100);
                sum += value;

                if (i == 0) {
                    max = value;
                    min = value;
                } else {
                    max = Math.max(max, value);
                    min = Math.min(min, value);
                }
            }

            SummaryDouble summary = SummaryDouble.newBuilder()
                .setCount(count)
                .setMax(max)
                .setMin(min)
                .setSum(sum)
                .build();

            TPoint point = TPoint.newBuilder()
                .setSummaryDouble(summary)
                .setTimestampsMillis(tsMillis)
                .build();

            points.add(point);

            tsMillis += INTERVAL_BETWEEN_POINTS_MILLIS;
        }

        return points;
    }
}
