package ru.yandex.solomon.experiments.rorewillo.pusher;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ExecutionException;

import com.google.common.primitives.Doubles;
import com.google.common.primitives.Longs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.model.protobuf.Histogram;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.client.ColumnFlagMask;

/**
 * @author Oleg Baryshnikov
 */
public class HistogramPusher {

    private static final Logger logger = LoggerFactory.getLogger(HistogramPusher.class);

    private static final long START_MILLIS = Instant.parse("2018-10-01T00:00:00Z").toEpochMilli();
    private static final long END_MILLIS = Instant.parse("2018-12-01T00:00:00Z").toEpochMilli();

    private static final int INTERVAL_BETWEEN_POINTS_MILLIS = 300_000;

    private static final Labels COMMON_LABELS = Labels.of(
        "project", "rorewillo-test",
        "cluster", "cluster",
        "service", "push_service",
        "version", "4");

    public static void main(String[] args) throws Exception {
        try (MetricPusher pusher = MetricPusher.create(logger)) {
            pushHistogram(pusher);
            System.exit(0);
        }
    }

    private static void pushHistogram(MetricPusher pusher) throws ExecutionException, InterruptedException {
        List<TPoint> points = generateHistogramPoints();

        saveAsPseudoHistogram(pusher, points);

        saveAsHistogram(pusher, points);
    }

    private static void saveAsHistogram(MetricPusher pusher, List<TPoint> points)
        throws InterruptedException, ExecutionException
    {
        Labels labels = COMMON_LABELS.toBuilder().add("sensor", "histogram_sensor").build();
        pusher.push(labels, MetricType.HIST, ColumnFlagMask.MINIMAL_HISTOGRAM, points).get();
    }

    private static void saveAsPseudoHistogram(MetricPusher pusher, List<TPoint> points) {
        Map<String, List<TPoint>> pointsByBound = new HashMap<>(10);

        for (TPoint point : points) {
            Histogram histogram = point.getHistogram();

            for (int bucketIdx = 0; bucketIdx < histogram.getBucketsCount(); ++bucketIdx) {
                long bucket = histogram.getBuckets(bucketIdx);
                double bound = histogram.getBounds(bucketIdx);

                TPoint rawPoint = TPoint.newBuilder()
                    .setDoubleValue(bucket)
                    .setTimestampsMillis(point.getTimestampsMillis())
                    .build();

                String boundName = Double.compare(bound, Histograms.INF_BOUND) == 0 ? "inf" : Double.toString(bound);

                if (!pointsByBound.containsKey(boundName)) {
                    pointsByBound.put(boundName, new ArrayList<>());
                }

                pointsByBound.get(boundName).add(rawPoint);
            }
        }

        for (Map.Entry<String, List<TPoint>> entry : pointsByBound.entrySet()) {
            String bin = entry.getKey();
            List<TPoint> binPoints = entry.getValue();

            Labels binLabels = COMMON_LABELS.toBuilder()
                .add("sensor", "pseudo_histogram_sensor")
                .add("bin", bin)
                .build();

            pusher.push(binLabels, MetricType.DGAUGE, ColumnFlagMask.MINIMAL_DOUBLE_MASK, binPoints);
        }
    }

    private static List<TPoint> generateHistogramPoints() {
        List<TPoint> points = new ArrayList<>();

        Random random = new Random(System.nanoTime());

        long tsMillis = START_MILLIS;

        double[] bounds = null;
        long[] buckets = null;

        while (tsMillis < END_MILLIS) {
            boolean canChangeBounds = random.nextInt(10000) > 9990;

            if (buckets == null || canChangeBounds) {
                int bucketsCount = random.nextInt(20) + 20;

                bounds = new double[bucketsCount];
                buckets = new long[bucketsCount];

                for (int i = 0; i < bucketsCount; ++i) {
                    bounds[i] = 10 + ((30000 - 10) * i / bucketsCount);
                }
            }

            for (int i = 0; i < bounds.length; ++i) {
                int maxBucket = (bounds.length - i) * 10;
                long bucket = random.nextInt(maxBucket);
                buckets[i] += bucket;
            }

            Histogram histogram = Histogram.newBuilder()
                .addAllBounds(Doubles.asList(bounds))
                .addAllBuckets(Longs.asList(buckets))
                .build();

            TPoint point = TPoint.newBuilder()
                .setHistogram(histogram)
                .setTimestampsMillis(tsMillis)
                .build();

            points.add(point);

            tsMillis += INTERVAL_BETWEEN_POINTS_MILLIS;
        }

        return points;
    }
}
