package ru.yandex.solomon.experiments.rorewillo.pusher;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.net.HostAndPort;
import org.slf4j.Logger;

import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientOptions;
import ru.yandex.metabase.client.MetabaseClients;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.metabase.api.protobuf.CreateOneRequest;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.tool.cfg.SolomonPorts;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TPoint;
import ru.yandex.stockpile.api.TWriteRequest;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientOptions;
import ru.yandex.stockpile.client.StockpileClients;
import ru.yandex.stockpile.client.StopStrategies;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class MetricPusher implements AutoCloseable {

    private static final long DEADLINE_MILLIS = 120_000;

    private final Logger logger;
    private final MetabaseClient metabase;
    private final StockpileClient stockpile;

    private MetricPusher(Logger logger, MetabaseClient metabase, StockpileClient stockpile) {
        this.logger = logger;
        this.metabase = metabase;
        this.stockpile = stockpile;
    }

    public static MetricPusher create(Logger logger) {
        return new MetricPusher(logger, metabaseClient(), stockpileClient());
    }

    private static StockpileClient stockpileClient() {
        List<HostAndPort> addresses = IntStream.range(0, 16)
            .mapToObj(n -> String.format("solomon-test-sas-%02d.search.yandex.net", n))
            .map(host -> HostAndPort.fromParts(host, SolomonPorts.STOCKPILE_GRPC))
            .collect(Collectors.toList());

        var options = StockpileClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setRequestTimeOut(1, TimeUnit.MINUTES)
                    .setKeepAliveDelay(1, TimeUnit.MINUTES)
                    .setKeepAliveTimeout(1, TimeUnit.SECONDS)
            )
            .setExpireClusterMetadata(30, TimeUnit.SECONDS)
            .setRetryStopStrategy(StopStrategies.stopAfterAttempt(20))
            .build();

        return StockpileClients.create(addresses, options);
    }

    private static MetabaseClient metabaseClient() {
        List<HostAndPort> addresses = IntStream.range(0, 16)
            .mapToObj(n -> String.format("solomon-test-sas-%02d.search.yandex.net", n))
            .map(host -> HostAndPort.fromParts(host, SolomonPorts.COREMON_GRPC))
            .collect(Collectors.toList());

        var options = MetabaseClientOptions.newBuilder(
                DefaultClientOptions.newBuilder()
                    .setRequestTimeOut(1, TimeUnit.MINUTES)
                    .setKeepAliveDelay(1, TimeUnit.MINUTES)
                    .setKeepAliveTimeout(1, TimeUnit.SECONDS))
            .setExpireClusterMetadata(30, TimeUnit.MINUTES)
            .setMetaDataRequestTimeOut(5, TimeUnit.MINUTES)
            .build();

        return MetabaseClients.create(addresses, options);
    }

    public CompletableFuture<Void> forceUpdate() {
        return metabase.forceUpdateClusterMetaData();
    }

    public CompletableFuture<Void> push(
        Labels labels,
        MetricType type,
        int mask,
        List<TPoint> points)
    {
        long deadlineMillis = System.currentTimeMillis() + DEADLINE_MILLIS;

        CreateOneRequest metabaseRequest = CreateOneRequest.newBuilder()
            .setMetric(Metric.newBuilder()
                .addAllLabels(LabelConverter.labelsToProtoList(labels))
                .setType(type)
                .build())
            .setDeadlineMillis(deadlineMillis)
            .build();

        logger.info("metabase request: " + metabaseRequest);

        return metabase.createOne(metabaseRequest)
            .thenCompose(metabaseResponse -> {
                if (metabaseResponse.getStatus() != EMetabaseStatusCode.OK) {
                    logger.error(
                        "not metabase ok: " + metabaseResponse.getStatus() + ": " + metabaseResponse
                            .getStatusMessage());
                    return CompletableFuture.completedFuture(null);
                }

                Metric metric = metabaseResponse.getMetric();

                MetricId metricId = metric.getMetricId();

                logger.info("Saving metrics: " + labels);

                TWriteRequest writeRequest = TWriteRequest.newBuilder()
                    .setMetricId(metricId)
                    .setColumnMask(mask)
                    .addAllPoints(points)
                    .setDeadline(deadlineMillis)
                    .build();

                logger.info("stockpile write request: " + ": " + metricId);

                return stockpile.writeOne(writeRequest)
                    .thenApply(stockpileResponse -> {
                        if (stockpileResponse.getStatus() != EStockpileStatusCode.OK) {
                            logger.error(
                                "not stockpile ok: " + stockpileResponse.getStatus() + ": "
                                    + stockpileResponse.getStatusMessage());
                        }

                        logger.info("success!");

                        return null;
                    });
            });
    }

    @Override
    public void close() {
        metabase.close();
        stockpile.close();
    }
}
