package ru.yandex.solomon.experiments.uranix;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;
import java.util.zip.GZIPOutputStream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.discovery.cluster.ClusterMapperImpl;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.metabase.client.MetabaseClient;
import ru.yandex.metabase.client.MetabaseClientFactory;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.config.SolomonConfigs;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayConfig;
import ru.yandex.solomon.config.thread.StubThreadPoolProvider;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.FindResponse;
import ru.yandex.solomon.metabase.api.protobuf.Metric;
import ru.yandex.solomon.metrics.client.Converters;
import ru.yandex.solomon.metrics.client.FindRequest;
import ru.yandex.solomon.model.protobuf.MetricId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricData;
import ru.yandex.stockpile.api.TCompressedReadManyResponse;
import ru.yandex.stockpile.api.TReadManyRequest;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.StockpileClientFactory;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class AlertStatusDownloader {
    private final MetabaseClient metabase;
    private final StockpileClient stockpile;

    private AlertStatusDownloader(String gatewayConfig) {
        TGatewayConfig config = SolomonConfigs.parseConfig(gatewayConfig, TGatewayConfig.getDefaultInstance());

        ThreadPoolProvider threadPool = new StubThreadPoolProvider();

        ClusterMapper clusterMapper = new ClusterMapperImpl(config.getClustersConfigList(), DiscoveryService.async(), threadPool.getExecutorService("", ""), threadPool.getSchedulerExecutorService());
        MetricRegistry registry = new MetricRegistry();

        var clientOptionsFactory = new ClientOptionsFactory(Optional.empty(), Optional.empty(), threadPool);
        var metabaseClientFactory = new MetabaseClientFactory(threadPool, clusterMapper, registry, clientOptionsFactory);
        var stockpileClientFactory = new StockpileClientFactory(threadPool, clusterMapper, registry, clientOptionsFactory);

        String cluster = config.getClustersConfig(0).getClusterId();
        System.out.println("Using cluster: " + cluster);

        metabase = metabaseClientFactory.createClients("alert-statuses-downloader", config.getMetabaseClientConfig()).get(cluster);
        stockpile = stockpileClientFactory.createClients("alert-statuses-downloader", config.getStockpileClientConfig()).get(cluster);
    }

    private long waitToLimitRate(int done, long nanosPassed, long limit, Duration perUnit) {
        double atLeastNanos = ((double)perUnit.toNanos()) / limit * done;
        return (long)(atLeastNanos - nanosPassed);
    }

    @JsonInclude(JsonInclude.Include.NON_NULL)
    private static class Row {
        public String alertId;
        @Nullable
        public String parentId;
        public String projectId;

        public int shardId;
        public long localId;
    }

    private void getMeta(Selectors selectors) throws IOException, InterruptedException {
        int offset = 0;
        int limitMetabase = 5000;

        int metabaseMetricsPerSecLimit = 10000;

        Map<Integer, List<Metric>> localIdByShardId = new HashMap<>();

        ObjectMapper mapper = new ObjectMapper();

        long startNanos = System.nanoTime();
        System.out.println("\nMETABASE\n");
        PrintStream ps = new PrintStream("/tmp/alerting_statuses.meta.tmp");
        while (true) {
            var range = findRange(selectors, offset, limitMetabase);
            if (range.isEmpty()) {
                break;
            }
            for (var metricKey : range) {
                MetricId stockpileKey = metricKey.getMetricId();
                Map<String, String> labels = LabelConverter.protoToLabels(metricKey.getLabelsList()).toMap();

                Row row = new Row();

                row.alertId = labels.get("alertId");
                row.parentId = labels.get("parentId");
                row.projectId = labels.get("projectId");

                row.shardId = stockpileKey.getShardId();
                row.localId = stockpileKey.getLocalId();

                ps.println(mapper.writeValueAsString(row));
            }
            offset += range.size();
            long elapsedNanos = System.nanoTime() - startNanos;
            long extraWaitNanos = Math.max(waitToLimitRate(offset, elapsedNanos, metabaseMetricsPerSecLimit, Duration.ofSeconds(1)), 0);

            Thread.sleep(extraWaitNanos / 1000000L, (int)(extraWaitNanos % 1000000L));
            double elapsedSecs = 1e-9 * (System.nanoTime() - startNanos);
            System.out.println(String.format("Resolving at %.2f metric/sec (waited %.6f s)", offset / elapsedSecs, extraWaitNanos / 1e9));
        }
        ps.close();
        System.out.println("Resolved " + offset + " metrics");
        Files.move(Path.of("/tmp/alerting_statuses.meta.tmp"), Path.of("/tmp/alerting_statuses.meta"), StandardCopyOption.REPLACE_EXISTING);
    }

    private void downloadShard(int shardId, List<Long> localIds, long fromMillis, long toMillis) {
        try {
            System.out.println("Shard " + shardId + ": starting");
            Path shardDir = Path.of("/home/uranix/alerting_statuses", shardId + ".pbzip");
            OutputStream os = new GZIPOutputStream(new FileOutputStream(shardDir.toFile()));
            int offset = 0;

            int limitStockpile = 50;
            int limitStockpileMin = 10;
            int limitStockpileMax = 200;

            long stockpileTime = 0;
            long writingTime = 0;

            final long SLEEP_MILLIS_DEFAULT = 2000;

            long sleepMillis = SLEEP_MILLIS_DEFAULT;

            while (offset < localIds.size()) {
                int offsetCommitted = offset;
                var request = TReadManyRequest.newBuilder()
                    .setShardId(shardId)
                    .setFromMillis(fromMillis)
                    .setToMillis(toMillis)
                    .setProducer(RequestProducer.STAFF)
                    .setBinaryVersion(StockpileFormat.CURRENT.getFormat());

                int high = Math.min(localIds.size(), offset + limitStockpile);
                for (; offset < high; offset++) {
                    request.addLocalIds(localIds.get(offset));
                }

                long tick = System.nanoTime();
                TCompressedReadManyResponse response = stockpile.readCompressedMany(request.build()).join();

                if (response.getStatus() != EStockpileStatusCode.OK) {
                    System.err.println("Shard " + shardId + ": " + response.getStatus());
                    offset = offsetCommitted;
                    Thread.sleep(sleepMillis);
                    sleepMillis *= 2;
                    limitStockpile = Math.max(limitStockpileMin, 6 * limitStockpile / 10);
                    continue;
                } else {
                    sleepMillis = SLEEP_MILLIS_DEFAULT;
                }

                stockpileTime += System.nanoTime() - tick;

                double stockpileSecsPerMetric = 1e-9 * stockpileTime / offset;
                limitStockpile = (int) (5 / stockpileSecsPerMetric);
                if (limitStockpile < limitStockpileMin) {
                    limitStockpile = limitStockpileMin;
                }
                if (limitStockpile > limitStockpileMax) {
                    limitStockpile = limitStockpileMax;
                }
                System.out.println("Shard " + shardId + ": adjusting request limit to " + limitStockpile);

                List<MetricData> metricsData = response.getMetricsList();

                tick = System.nanoTime();
                for (MetricData metricData : metricsData) {
                    metricData.writeDelimitedTo(os);
                }
                writingTime += System.nanoTime() - tick;
                System.out.printf("Shard " + shardId + ": Timings: stockpile: %.3f us/metric, fs: %.3f us/metric\n", stockpileTime / 1e3 / offset, writingTime / 1e3 / offset);
            }
            os.close();
        } catch (Throwable t) {
            t.printStackTrace();
            throw new RuntimeException(t);
        }
    }

    private void run(Selectors selectors) throws InterruptedException, IOException {
        ObjectMapper mapper = new ObjectMapper();
        HashMap<Integer, List<Row>> rowsByShardId = new HashMap<>();
        while (true) {
            try {
                BufferedReader br = new BufferedReader(new FileReader("/tmp/alerting_statuses.meta"));
                String s;
                while ((s = br.readLine()) != null) {
                    Row row = mapper.readValue(s, Row.class);
                    List<Row> shard = rowsByShardId.computeIfAbsent(row.shardId, ignore -> new ArrayList<>());
                    shard.add(row);
                }
                break;
            } catch (FileNotFoundException e) {
                getMeta(selectors);
            }
        }

        System.out.println("Data spans across " + rowsByShardId.size() + " stockpile shards");
        for (var line : rowsByShardId.entrySet()) {
            System.out.println(line.getKey() + ": " + line.getValue().size() + " metrics");
        }

        System.out.println("\nSTOCKPILE\n");

        long toMillis = Instant.now().toEpochMilli();
        long fromMillis = toMillis - Duration.ofDays(14).toMillis();

        ExecutorService executor = Executors.newFixedThreadPool(20);
        List<CompletableFuture<Void>> futures = new ArrayList<>();

        for (var line : rowsByShardId.entrySet()) {
            int shardId = line.getKey();
            List<Long> localIds = line.getValue().stream().map(row -> row.localId).collect(Collectors.toList());

            futures.add(CompletableFuture.supplyAsync(() -> {
                downloadShard(shardId, localIds, fromMillis, toMillis);
                return null;
            }, executor));
        }

        CompletableFutures.allOfVoid(futures).join();

        executor.shutdown();
    }

    private List<Metric> findRange(Selectors selectors, int offset, int limit) {
        FindResponse response = metabase.find(Converters.toProto(FindRequest.newBuilder()
            .setSelectors(selectors)
            .setOffset(offset)
            .setLimit(limit)
            .setProducer(RequestProducer.REQUEST_PRODUCER_UNSPECIFIED)
            .build())
        ).join();

        if (response.getStatus() != EMetabaseStatusCode.OK) {
            throw new RuntimeException(response.getStatus().toString());
        }

        return response.getMetricsList();
    }

    public static void main(String[] args) {
        Selectors selectors = Selectors.parse("{project='solomon', cluster='production', service='alerting_statuses', sensor='alert.evaluation.status'}");
        //Selectors selectors = Selectors.parse("{project='solomon', cluster='prestable', service='sys_self'}");

        String gatewayConfig = "/home/uranix/arcadia/solomon/configs/production/gateway.conf";
        //String gatewayConfig = "/home/uranix/arcadia/solomon/configs/prestable/gateway.conf";

        try {
            (new AlertStatusDownloader(gatewayConfig)).run(selectors);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            System.exit(0);
        }
    }
}
